"rqpd.fit" <- function (X, Z, y, taus, tauw, lambda, control, method = "sfn", ...)
{
    N <- length(y)
    p <- ncol(X)
    n <- ncol(Z)
    K <- length(tauw)
    Fidelity <- cbind(as(tauw,"matrix.diag.csr") %x% X,tauw %x% Z)
    Penalty <- cbind(as.matrix.csr(0,n,K*p),lambda*as(n,"matrix.diag.csr"))
    D <- rbind(Fidelity,Penalty)
    y <- c(tauw %x% y,rep(0,n))
    rhs <- c((tauw*(1-taus)) %x% (t(X)%*%rep(1,N)),
       sum(tauw*(1-taus)) * (t(Z) %*% rep(1,N)) + lambda * rep(1/2,n))
    # Thus far no constrained form, but un bel di ...
    f <- switch(method,
	sfn = rq.fit.sfn(D, y, rhs = rhs, control = control, ...),
        sfnc = rq.fit.sfnc(D, y, rhs = rhs, control = control, ...), {
            what <- paste("rq.fit.", method, sep = "")
            if (exists(what, mode = "function"))
                (get(what, mode = "function"))(X, y, ...)
            else stop(paste("unimplemented method:", method))
        })
    if((f$ierr != 0) && (f$it < 5)) 
        warning(paste("Dubious convergence:",sfnMessage(f$ierr)))
    fit <- list(coefficients = f$coef, ierr = f$ierr, it = f$it)
    fit$contrasts <- attr(X, "contrasts")
    fit$resid <- c(y - D %*% fit$coef)
    fit
}

"rqpd" <- function (formula, taus = 1:3/4, data = parent.frame(), 
	tauw = c(.25,.5,.25), na.action, lambda = 1, contrasts = NULL, 
    	ztol = 1e-05,  control = NULL, ...) 
{
    ## Process Call
    call <- match.call()
    m <- match.call(expand = FALSE)
    temp <- c("", "formula", "data", "tauw", "na.action")
    if(!(length(taus) == length(tauw))) stop("lengths of taus and w  don't match") 
    m <- m[match(temp, names(m), nomatch = 0)]
    m[[1]] <- as.name("model.frame")

    ## Formula
    require(Formula)
    oformula <- as.formula(formula)
    formula <- as.Formula(formula)
    if(length(formula)[2] < 2L) {
      formula <- as.Formula(formula(formula), formula(formula, lhs = 0L))
    } else {
      if(length(formula)[2] > 2L) {
        formula <- Formula(formula(formula, rhs = 1:2))
        warning("formula shouldn't have more than two RHS parts")
      }
    }
    m$formula <- formula

    ## Evaluate model.frame
    m[[1L]] <- as.name("model.frame")
    m <- eval(m, parent.frame())

    ## Extract terms, model matrix and response
    require(MatrixModels)
    mt <- terms(formula, data = data)
    mtX <- terms(formula, data = data, rhs = 1L)
    mtZ <- delete.response(terms(formula, data = data, rhs = 2L))
    attr(mtZ, "intercept") <- 0L
    y <- model.response(m, "any")
    X <- model.Matrix(mtX, m, contrasts, sparse = TRUE)
    Xnames <- dimnames(X)[[2]]
    X <- as(X ,"matrix.csr")
    Z <- model.Matrix(mtZ, m, sparse = TRUE)
    Znames <- dimnames(Z)[[2]]
    Z <- as(Z ,"matrix.csr")
    ids <- m[,attr(mtZ,"term.labels")]

    ## Check for Singularity of X 
    if (det(t(X) %*% X) < ztol) stop("Singular design matrix")

    ## Set default control parameters
    if(is.null(control)) control <- sfn.control(warn.mesg = FALSE)

    f <- rqpd.fit(X, Z, y, taus, tauw, lambda, method = "sfn", control = control, ...)
    fit <- list(coefficients = f$coef, residuals = f$resid)
    names(fit$coef) <- c(paste(rep(Xnames,length(taus)),"[",
        rep(taus,each=length(Xnames)),"]",sep = ""),Znames)
    fit$X <- X
    fit$Z <- Z
    fit$y <- y
    fit$taus <- taus
    fit$tauw <- tauw
    fit$lambda <- lambda
    fit$control <- control
    fit$ids <- ids
    fit$formula <- oformula
    fit$call <- call
    fit$rank <- sum(abs(f$coef) > ztol)
    attr(fit, "na.message") <- attr(m, "na.message")
    class(fit) <- "rqpd"
    fit
}
"print.rqpd" <-
function(x, ...)
{
	if(!is.null(cl <- x$call)) {
		cat("Call:\n")
		dput(cl)
	}
	coef <- coef(x)
	cat("\nCoefficients:\n")
	print(coef, ...)
	rank <- x$rank
	nobs <- length(x$y)
	rdf <- nobs - rank
	cat("\nDegrees of freedom:", nobs, "total;", rdf, "residual\n")
	if(!is.null(attr(x, "na.message")))
		cat(attr(x, "na.message"), "\n")
	invisible(x)
}
"print.summary.rqpd" <-
function(x, digits = max(5, .Options$digits - 2), ...)
{
	cat("\nCall: ")
	dput(x$call)
	coef <- x$coef
	taus <- x$taus
	tauw <- x$tauw
	cat("\ntaus: ")
	print(format(round(taus, digits = digits)), quote = FALSE, ...)
	cat("\ntau weights: ")
	print(format(round(tauw, digits = digits)), quote = FALSE, ...)
	cat("\nCoefficients:\n")
	print(format(round(coef, digits = digits)), quote = FALSE, ...)
	invisible(x)
}




# Routine for bootstrapping rqpd models
# On Input:
#	ids ~ factor variable with ids of fixed effect "individuals"
#	X ~ systematic portion of the design matrix in .csr form
#	Z ~ Fixed effect portion of the design matrix in .csr form
#	y ~ response vector
#	taus ~ vector of taus
#	tauw ~ vector of weights for taus
# Techniques:
#	1. "xy"  method samples on "individuals"
#	2. "wxy" method weights on "individuals"
#
"boot.rqpd"<-
function (ids, X, Z, y, taus = 1:3/4, tauw = c(.25,.5,.25), lambda, control, 
	R = 200, bsmethod = "xy", mofn = length(y), ...)
{
    n <- length(y)
    p <- ncol(X) * length(taus) + ncol(Z)
    L <- length(levels(as.factor(ids)))
    M <- table(ids)
    B <- matrix(0, R, p)
    gsample <- function(ids, m, L = length(levels(as.factor(ids)))){
           g <- split(1:length(ids),ids)
           unlist(g[sample(L,m,replace = TRUE)],use.names = FALSE)
           }
    if (bsmethod == "xy") {
	if(mofn < p || mofn > n) stop("mofn is out of range")
        for(i in 1:R){
             #s <- gsample(ids, mofn, L) # VERY SLOW due to "["
             #B[i,] <- rqpd.fit(X[s,],Z[s,],y[s],taus,tauw, lambda, control, ...)$coef
             w <- rep(rmultinom(1,L,rep(1/L,L)),M)
             B[i,] <- rqpd.fit(w*X,w*Z,w*y,taus,tauw, lambda, control, ...)$coef
             }   
         B <- sqrt(mofn/n)*B
         }
    else if (bsmethod == "wxy") {
        for(i in 1:R){
             w <- rep(rexp(L,1),M)
             B[i,] <- rqpd.fit(w*X,w*Z,w*y,taus,tauw, lambda, control, ...)$coef
             }
        }   
    else stop("Your specified bootstrap method is not implemented")
    B
}
"summary.rqpd" <- function (object, se = "boot", covariance = FALSE, ...)
{
    X <- object$X
    Z <- object$Z
    y <- object$y
    ids <- object$ids
    taus <- object$taus
    tauw <- object$tauw
    lambda <- object$lambda
    control <- object$control
    coef <- object$coef
    cnames <- names(coef)
    resid <- object$residuals
    rank <- object$rank
    n <- length(y)
    p <- length(coef)
    rdf <- n - rank
    if(NCOL(ids) > 1) stop("Only one-way layouts for ids allowed for summary")
    if (se == "boot") {
        B <- boot.rqpd(ids, X, Z, y, taus, tauw, lambda, control, ...)
        cov <- cov(B)
        serr <- sqrt(diag(cov))
    }
    else stop("Only boot method for rqpd objects")
    coef <- array(coef, c(p, 4))
    dimnames(coef) <- list(cnames, c("Value", "Std. Error", "t value", "Pr(>|t|)"))
    coef[, 2] <- serr
    coef[, 3] <- coef[, 1]/coef[, 2]
    coef[, 4] <- if(rdf > 0) 2 * (1 - pt(abs(coef[, 3]), rdf)) else NA
    object <- object[c("call")]
    if (covariance == TRUE) object$cov <- cov 
    object$coefficients <- coef
    object$rdf <- rdf
    object$taus <- taus
    object$tauw <- tauw
    object$lambda <- lambda
    class(object) <- "summary.rqpd"
    object
}

