"rqss.fit" <-
function (x, y, tau = 0.5, method = "sfn", ...)
{
    fit <- switch(method,
	sfn = rq.fit.sfn(x, y, tau = tau, ...),
        sfnc = rq.fit.sfnc(x, y, tau = tau, ...), {
            what <- paste("rq.fit.", method, sep = "")
            if (exists(what, mode = "function"))
                (get(what, mode = "function"))(x, y, ...)
            else stop(paste("unimplemented method:", method))
        })
    fit$contrasts <- attr(x, "contrasts")
    fit$resid <- c(y - x %*% fit$coef)

    fit
}
"untangle.specials" <-
function (tt, special, order = 1)
{
    spc <- attr(tt, "specials")[[special]]
    if (length(spc) == 0)
        return(list(vars = character(0), terms = numeric(0)))
    facs <- attr(tt, "factor")
    fname <- dimnames(facs)
    ff <- apply(facs[spc, , drop = FALSE], 2, sum)
    list(vars = (fname[[1]])[spc], terms = seq(ff)[ff & match(attr(tt,
        "order"), order, nomatch = 0)])
}
"qss" <-
function (x, constraint = "N", lambda = 1, ndum = 0, dummies = NULL, w = rep(1, length(x)))
{
    if(is.matrix(x)){
       if (ncol(x) == 2)
           qss <- qss2(x, constraint = constraint, dummies = dummies,
                lambda = lambda, ndum = ndum, w = w)
       else if(ncol(x) == 1)
           x <- as.vector(x)
       else
           stop("qss objects must have dimension 1 or 2")
        }
    if(is.vector(x))
       qss <- qss1(x, constraint = constraint, lambda = lambda,
            dummies = dummies, ndum = ndum, w = w)
    qss
}

"qss2" <-
function(x, y, constraint = "N", lambda = 1, ndum= 0, dummies = NULL, w=rep(1,length(x))){
#
# Sparse Additive Quantile Smoothing Spline Models - Bivariate (Triogram) Module
#
# This function returns a structure intended to make model.matrix for a bivariate
# nonparametric component of a model formula specified by a call to rqss().  A sparse form
# of the Frisch Newton algorithm is eventually called to compute the estimator.
# An optional convexity/concavity constraint can be specified.  If
# the formula consists of a single qss component then the estimator solves the
# following variational problem:
#
#       min sum rho_tau (y_i - g(x_i)) + lambda V(grad(g))
#
# where V(f) denotes the total variation of the function f.  The solution is a piecewise
# linear function on the Delaunay triangulation formed by the observed (x_i,y_i) points.
# Additive models can consist
# of several components of this form plus partial linear and univariate qss components.
# To resolve the identifiability problem we delete the first column of the qss design
# components.  On return F contains the fidelity portion of the design, A the penalty
# contribution of the design, R the constraint portion, and r the rhs of the constraints.
#
# Constraints are specified by the constraint argument:
#
#	N	none
#	U	convex
#	C	concave
#
# Author:  Roger Koenker   April 2, 2003
#
# For a prototype see triogram in ~roger/projects/tv/cobar/.RData on ysidro.
#
# Warning:   Under development...todo:
#
#       o  weights
#       o  dummy x's
#       o  tau's
#       o  lambda's
#       o  ...
#
   require(tripack)
#
    y <- x[,2]
    x <- x[,1]
    n <- length(x)
    if (n != length(y))
        stop("xy lengths do not match")
    f <- triogram.fidelity(x, y, ndum = ndum, dummies = dummies)
    F <- f$F
    A <- triogram.penalty(f$x, f$y)
    switch(constraint, V = {
        R <- A
        r <- rep(0, nrow(R))
    }, C = {
        R <- -A
        r <- rep(0, nrow(R))
    }, N = {
        R = NULL
        r = NULL
    })
    list(x = list(x = f$x, y = f$y), F = F[, -1], dummies = f$dummies,
        lambda = lambda, A = A[, -1], R = R[, -1], r = r)
}


"qss1" <-
function (x, constraint = "N", lambda = 1, dummies = dummies,
        ndum = 0, w = rep(1, length(x))){
#
# Sparse Additive Quantile Smoothing Spline Models - Univariate Module
#
# This function returns a structure intended to make model.matrix for a univariate
# nonparametric component of a model formula specified by a call to rq().  A sparse form
# of the Frisch Newton algorithm is eventually called to compute the estimator.
# Optional monotonicity and/or convexity/concavity constraints can be specified.  If
# the formula consists of a single qss component then the estimator solves the
# following variational problem:
#
#       min sum rho_tau (y_i - g(x_i)) + lambda V(g')
#
# where V(f) denotes the total variation of the function f.  The solution is a piecewise
# linear function with "knots" at the observed x_i points.  Additive models can consist
# of several components of this form plus partial linear and triogram components.
# To resolve the identifiability problem we delete the first column of the qss design
# components.  On return F contains the fidelity portion of the design, A the penalty
# contribution of the design, R the constraint portion, and r the rhs of the constraints.
#
# Constraints are specified by the constraint argument:
#
#	N	none
#	I	monotone increasing
#	D	monotone decreasing
#	V	convex
#	C	concave
#	CI	concave and monotone increasing
#	...	etc
#
# Author:  Roger Koenker   February 27, 2003
#
# Warning:   Under development...todo:
#
#       o  weights
#       o  dummy x's
#       o  tau's
#       o  lambda's
#       o  ...
#
#
    xun <- unique(x[order(x)])
    h <- diff(xun)
    nh <- length(h)
    nx <- length(x)
    p <- nh + 1
    B <- new("matrix.csr", ra = c(rbind(-1/h, 1/h)), ja = as.integer(c(rbind(1:nh,
        2:(nh + 1)))), ia = as.integer(2 * (1:(nh + 1)) - 1),
        dimension = as.integer(c(nh, nh + 1)))
    makeD <- function(p) {
        new("matrix.csr", ra = c(rbind(rep(-1, (p - 1)), rep(1,
            (p - 1)))), ja = as.integer(c(rbind(1:(p - 1), 2:p))),
            ia = as.integer(2 * (1:p) - 1), dimension = as.integer(c(p -
                1, p)))
    }
    D <- makeD(nh)
    A <- D %*% B
    if (length(xun) == length(x)){
        F <- new("matrix.csr", ra = rep(1, nx), ja = as.integer(rank(x)),
            ia = 1:(nx + 1), dimension = as.integer(c(nx, nx)))
	}
    else {
        F <- new("matrix.csr", ra = rep(1, nx), ja = as.integer(factor(x)),
            ia = 1:(nx + 1), dimension = as.integer(c(nx, length(xun))))
	}

   switch(constraint,
        V = {   R <- A;
                r <- rep(0,nrow(R))
                },
        C = {   R <- -A;
                r <- rep(0,nrow(R))
                },
        I = {   R <- makeD(p)
                r <- rep(0,p-1)
                },
        D = {   R <- -makeD(p)
                r <- rep(0,p-1)
                },
        VI = {  R <- makeD(p)
		R <- rbind(R,A)
                r <- rep(0,nrow(R))
		},
        VD = {  R <- -makeD(p)
		R <- rbind(R,A)
                r <- rep(0,nrow(R))
		},
        CI = {  R <- makeD(p)
		R <- rbind(R,-A)
                r <- rep(0,nrow(R))
		},
        CD = {  R <- -makeD(p)
		R <- rbind(R,-A)
                r <- rep(0,nrow(R))
		},
	N = { R=NULL; r=NULL}
	)
   list(x = list(x=xun), F=F[,-1], lambda = lambda, A=A[,-1], R=R[,-1], r=r)
}
"plot.qss1" <-
function(x, add = FALSE, ...)
{
if(!add) plot(x[,1],x[,2],type = "n", ...)
lines(x[,1],x[,2], ...)
}
"plot.qss2" <-
function (x, render = "contour", ncol = 100, zcol = NULL, ...)
{
    require(tripack)
    y <- x[, 2]
    z <- x[, 3]
    x <- x[, 1]
    tri <- tri.mesh(x, y)
    if (render == "rgl") {
        if(!require("rgl",quietly=TRUE))
		stop("The package rgl is missing")
        collut <- terrain.colors(ncol)
        if (!length(zcol))
            zcol <- z
        if (max(z) > max(zcol) || min(z) < min(zcol))
            warning("fitted z values out of range of zcol vector")
        zlim <- range(zcol)
        colz <- ncol * (z - zlim[1])/(zlim[2] - zlim[1]) + 1
        colz <- collut[colz]
        s <- c(t(triangles(tri)[, 1:3]))
        rgl.triangles(x[s], y[s], z[s], col = colz[s])
    }
    else {
        require(akima)
        if(render == "contour"){
                plot(x, y, type = "n", ...)
                contour(interp(x, y, z), add = TRUE, frame.plot = TRUE, ...)
                convex.hull(tri, plot.it = TRUE, add = TRUE)
                }
        else if(render == "persp")
                persp(interp(x, y, z, ), theta = -40, phi = 20, xlab = "x",
                        ylab = "y", zlab = "z", ...)
        else stop(paste("Unable to render: ",render))
    }
}
"plot.rqss" <-
function (x, ...)
{
    m <- length(x$qss)
    if(m > 1) par(ask = TRUE)
    qssnames <- names(x$qss)
    for (i in 1:m) {
        qss <- x$qss[[i]]$xyz
        if (ncol(qss) == 3) {
            qss[, 3] <- x$coef[1] + qss[, 3]
            plot.qss2(qss,  ...)
        }
        else if (ncol(qss) == 2) {
            qss[, 2] <- x$coef[1] + qss[, 2]
            plot.qss1(qss, xlab = paste(qssnames[i]), ylab = "Effect", ...)
            title(paste("Effect of ", qssnames[i]))
        }
        else stop("invalid fitted qss object")
    }
    par(ask = FALSE)
}
"triogram.fidelity" <- function (x, y, ndum=0, dummies = NULL)
{
#Make fidelity block of the triogram design in sparse matrix.csr form
#The rather esoteric match call identifies and handles duplicated xy points
n <- length(x)
A <- as.data.frame(cbind(x,y))
dupA <- duplicated(A)
if(any(dupA)){
  x <- x[!dupA]
  y <- y[!dupA]
  J <- match(do.call("paste",c(A,"\r")),do.call("paste",c(A[!dupA,],"\r")))
  z <- new("matrix.csr",ra=rep(1,n), ja=J, ia=1:(n+1),dimension=as.integer(c(n,max(J))))
  }
else{
  z <- as(n,"matrix.diag.csr")
  z <- as(z,"matrix.csr")
 }
#Augment with dummy vertices, if any...
if(length(dummies)){
	if (is.list(dummies)){
        	if (all(!is.na(match(c("x", "y"), names(dummies))))){
        		ndum <- length(dummies$x)
			if(length(dummies$y) == ndum){
        			x <- c(x,dummies$x)
        			y <- c(y,dummies$y)
        			zdum <- as.matrix.csr(0,n,ndum)
        			z <- cbind(z,zdum)
				}
        		else stop("dummies x and y components differ in length")
			}
        	else stop("dummies list lacking x and y elements")
		}
    	else stop("dummies argument invalid (not a list) in triogram.fidelity")
	}
else if(ndum > 0){
        u <- runif(ndum); v <- runif(ndum)
        xd <- min(x) + u * (max(x)-min(x))
        yd <- min(y) + v * (max(y)-min(y))
        T <- tri.mesh(x,y)
        s <- in.convex.hull(T,xd,yd)
        x <- c(x,xd[s])
        y <- c(y,yd[s])
        ndum <- sum(s)
        zdum <- as.matrix.csr(0,n,ndum)
        z <- cbind(z,zdum)
	dummies <- list(x = xd[s],y = yd[s])
        }
list(x=x,y=y,F=z, dummies = dummies)
}
"triogram.penalty" <- function (x, y, eps = .Machine$double.eps)
{
    n <- length(x)
    tri <- tri.mesh(x, y)
    bnd <- on.convex.hull(tri,x,y)
    q <- length(tri$tlist)
    m <- 13 * n
    z <- .Fortran("penalty", as.integer(n), as.integer(m), as.integer(q),
        as.double(x), as.double(y), as.integer(bnd),as.integer(tri$tlist),
        as.integer(tri$tlptr), as.integer(tri$tlend), rax = double(m),
	jax = integer(m), ned = integer(1), as.double(eps), ierr = integer(1),
	PACKAGE = "quantreg")[c("rax", "jax", "iax", "ned", "ierr")]
    if (z$ierr == 1)
        stop("collinearity in ggap")
    nnz <- 4 * z$ned
    ra <- z$rax[1:nnz]
    ja <- z$jax[1:nnz]
    ia <- as.integer(1 + 4 * (0:z$ned))
    dim <- as.integer(c(z$ned, n))
    new("matrix.csr",ra=ra,ja=ja,ia=ia,dimension=dim)
}
"predict.rqss" <-
function (object, newdata,  ...)
{
    Terms <- delete.response(terms(object$formula, "qss"))
    Names <- all.vars(Terms)
    if(any(!(Names %in% names(newdata))))
        stop("newdata doesn't include some model variables")
    ff <- reformulate(all.vars(Terms))
    nd <- eval(model.frame(ff,data=newdata),parent.frame())
    qssterms <- attr(Terms,"specials")$qss
    if(length(qssterms)){
        tmp <- untangle.specials(Terms, "qss")
        dropv <- tmp$terms
        m <- length(dropv)
        if(length(dropv))
            PLTerms <- Terms[-dropv]
        attr(PLTerms, "specials") <- tmp$vars
        }
    else {
        PLTerms <- Terms
	m <- 0
	}
    X <- model.matrix(PLTerms, data = nd)  #FIXME  Factors/Contrasts need work!
    p <- ncol(X)
    y <- X %*% object$coef[1:p]
    if(m > 0) {
        for (i in 1:m) {
            qss <- object$qss[[i]]$xyz
            names <- all.vars(Terms[dropv[i]])
            dimnames(qss)[[2]] <- c(names,"zfit")
            newd <- nd[names]
            if (ncol(qss) == 3) {
                y <- y + predict.qss2(qss,newdata = newd, ...)$z
            }
            else if (ncol(qss) == 2) {
                y <- y + predict.qss1(qss,newdata = newd, ...)$y
            }
            else stop("invalid fitted qss object")
           }
        }
    y
}

"predict.qss1" <-
function (object, newdata, ...)
{
    x <- object[, 1]
    y <- object[, 2]
    if(ncol(newdata)==1)
        newdata <- newdata[,1]
    else
        stop("newdata should have only one column for predict.qss1")
    if (any(diff(x) < 0))
        stop("x coordinates in object not monotone")
    if (max(newdata) >= max(x) || min(newdata) <= min(x))
        stop("no extrapolation allowed in predict.qss")
    bin <- cut(newdata, unique(x), label = FALSE)
    p <- length(x)
    m <- length(newdata)
    V <- cbind(bin, bin + 1)
    B <- cbind(x[bin + 1] - newdata, newdata - x[bin])/(x[bin +
        1] - x[bin])
    ra <- c(t(B))
    ja <- as.integer(c(t(V)))
    ia <- as.integer(c(2 * (1:m) - 1, 2 * m + 1))
    dim <- c(m, p)
    D <- new("matrix.csr", ra = ra, ja = ja, ia = ia, dimension = dim)
    list(x = newdata, y = D %*% y)
}

predict.qss2 <- function (object, newdata, ...) 
{
    x <- object[, 1]
    y <- object[, 2]
    z <- object[, 3]
    tri.area <- function(v) {
        0.5 * ((v[2, 1] - v[1, 1]) * (v[3, 2] - v[1, 2]) - (v[3, 
            1] - v[1, 1]) * (v[2, 2] - v[1, 2]))
    }
    barycentric <- function(v) {
        b <- rep(0, 3)
        Area <- tri.area(v[1:3, ])
        b[1] <- tri.area(v[c(4, 2, 3), ])/Area
        b[2] <- tri.area(v[c(1, 4, 3), ])/Area
        b[3] <- tri.area(v[c(1, 2, 4), ])/Area
        if (any(b < 0 || b > 1)) 
            stop("barycentric snafu")
        b
    }
    if (is.list(newdata)) {
	fnames <- (dimnames(object)[[2]])[1:2]
        if (all(!is.na(match(fnames, names(newdata))))) {
            newx <- newdata[[fnames[1]]]
            newy <- newdata[[fnames[2]]]
	   }
        else (stop("qss object and newdata frame names conflict"))
        }
    else if (is.matrix(newdata)) 
        if (ncol(newdata) == 2) {
            newx <- newdata[, 1]
            newy <- newdata[, 2]
        }
        else (stop("newdata matrix must have 2 columns"))
    tri <- tri.mesh(x, y)
    if (!all(in.convex.hull(tri, newx, newy))) 
        stop("some newdata points outside convex hull")
    p <- length(x)
    m <- length(newx)
    V <- matrix(0, m, 3)
    B <- matrix(0, m, 3)
    for (i in 1:m) {
        V[i, ] <- unlist(tri.find(tri, newx[i], newy[i]))
        v <- rbind(cbind(x[V[i, ]], y[V[i, ]]), c(newx[i], newy[i]))
        B[i, ] <- barycentric(v)
    }
    ra <- c(t(B))
    ja <- as.integer(c(t(V)))
    ia <- as.integer(3 * (0:m) + 1)
    D <- new("matrix.csr", ra = ra, ja = ja, ia = ia, dimension = c(m, 
        p))
    list(x = newx, y = newy, z = c(D %*% z))
}
"rqss" <- function (formula, tau = 0.5, data = parent.frame(), weights, 
    na.action, method = "sfn", lambda = NULL, contrasts = NULL, ...) 
{
    call <- match.call()
    m <- match.call(expand = FALSE)
    temp <- c("", "formula", "data", "weights", "na.action")
    m <- m[match(temp, names(m), nomatch = 0)]
    m[[1]] <- as.name("model.frame")
    special <- "qss"
    Terms <- if (missing(data)) 
        terms(formula, special)
    else terms(formula, special, data = data)
    qssterms <- attr(Terms, "specials")$qss
    dropx <- NULL
    if (length(qssterms)) {
        tmpc <- untangle.specials(Terms, "qss")
        ord <- attr(Terms, "order")[tmpc$terms]
        if (any(ord > 1)) 
            stop("qss can not be used in an interaction")
        dropx <- tmpc$terms
        if (length(dropx)) 
            Terms <- Terms[-dropx]
        attr(Terms, "specials") <- tmpc$vars
        qssnames <- unlist(lapply(parse(text = tmpc$vars), function(x) deparse(x[[2]])))
    }
    m$formula <- Terms
    m <- eval(m, parent.frame())
    weights <- model.extract(m, weights)
    process <- (tau < 0 || tau > 1)
    Y <- model.extract(m, "response")
    X <- model.matrix(Terms, m, contrasts)
    p <- ncol(X)
    pf <- environment(formula)
    if(method == "lasso"){
	    if(!length(lambda))
		stop("No lambda specified for lasso constraint")
	    if(length(lambda) == 1)
		lambda <- c(0,rep(lambda,p-1))
	    if(length(lambda) != p)
		stop("lambda must be either of length p, or length 1")
	    if(any(lambda < 0))
		stop("negative lambdas disallowed")
	    L <- diag(lambda,nrow = length(lambda))
	    L <- L[which(lambda != 0), , drop = FALSE]
            L <- as.matrix.csr(L)
	}
# Now make the rest of the fidelity matrix, sparsify and append.
    if (length(qssterms) > 0) {
        F <- as.matrix.csr(X)
        qss <- lapply(tmpc$vars, function(u) eval(parse(text = u), 
            data, enclos = pf))
        mqss <- length(qss)
        ncA <- rep(0, mqss + 1)
        nrA <- rep(0, mqss + 1)
        nrR <- rep(0, mqss + 1)
        for (i in 1:mqss) {
            F <- cbind(F, qss[[i]]$F)
            ncA[i + 1] <- ncol(qss[[i]]$A)
            nrA[i + 1] <- nrow(qss[[i]]$A)
            nrR[i + 1] <- ifelse(is.null(nrow(qss[[i]]$R)), 0, 
                nrow(qss[[i]]$R))
        }
        A <- as.matrix.csr(0, sum(nrA), sum(ncA))
        if (sum(nrR) > 0) {
            R <- as.matrix.csr(0, sum(nrR), sum(ncA))
            nrR <- cumsum(nrR)
        }
        ncA <- cumsum(ncA)
        nrA <- cumsum(nrA)
        lambdas <- rep(0, mqss)
        for (i in 1:mqss) {
            lambdas[i] <- qss[[i]]$lambda
            Arows <- (1 + nrA[i]):nrA[i + 1]
            Acols <- (1 + ncA[i]):ncA[i + 1]
            A[Arows, Acols] <- qss[[i]]$lambda * qss[[i]]$A
            if (nrR[i] < nrR[i + 1]) 
                R[(1 + nrR[i]):nrR[i + 1], (1 + ncA[i]):ncA[i + 
                  1]] <- qss[[i]]$R
        }
        A <- cbind(as.matrix.csr(0, nrA[mqss + 1], p), A)
        if (nrR[mqss + 1] > 0) {
            R <- cbind(as.matrix.csr(0, nrR[mqss + 1], p), R)
            r <- rep(0, nrR[mqss + 1])
        }
        else {
            R <- NULL
            r <- NULL
        }
        if(method == "lasso"){
	    A <- rbind(cbind(L,as.matrix.csr(0,nrow(L), ncol(F)-ncol(L))),A)
	    }
        X <- rbind(F, A)
        Y <- c(Y, rep(0, nrow(A)))
        rhs <- t(rbind((1 - tau) * F, 0.5 * A)) %*% rep(1, nrow(X))
        XpX <- t(X) %*% X
        nnzdmax <- XpX@ia[length(XpX@ia)] - 1
        nsubmax <- max(nnzdmax, floor(1000 + exp(-1.6) * nnzdmax^1.2))
        nnzlmax <- floor(2e+05 - 2.8 * nnzdmax + 7e-04 * nnzdmax^2)
        tmpmax <- floor(1e+05 + exp(-12.1) * nnzdmax^2.35)
        fit <- if (length(r) > 0) 
            rqss.fit(X, Y, tau = tau, rhs = rhs, method = "sfnc", 
                R = R, r = r, nsubmax = nsubmax, nnzlmax = nnzlmax, 
                tmpmax = tmpmax)
        else rqss.fit(X, Y, tau = tau, rhs = rhs, method = "sfn", 
            nsubmax = nsubmax, nnzlmax = nnzlmax, tmpmax = tmpmax)
        for (i in 1:mqss) {
            ML <- p + 1 + ncA[i]
            MU <- p + ncA[i + 1]
            qss[[i]] <- list(xyz = cbind(qss[[i]]$x$x, qss[[i]]$x$y, 
                c(0, fit$coef[ML:MU])), dummies = qss[[i]]$dummies)
        }
        names(qss) <- qssnames
        fit$qss <- qss
    }
    else {
        if(method == "lasso")
             X <- rbind(X,L)
        fit <- if (length(weights)) 
            rq.wfit(X, Y, tau = tau, weights, method, ...)
        else rq.fit(X, Y, tau = tau, method, ...)
    }
    fit$terms <- Terms
    fit$formula <- formula
    fit$call <- call
    fit$tau <- tau
    if (length(qssterms)) {
        fit$lambdas <- lambdas
        fit$nrA <- nrA
    }
    else fit$lambdas <- fit$nrA <- NA
    attr(fit, "na.message") <- attr(m, "na.message")
    class(fit) <- "rqss"
    fit
}
"summary.rqss" <- function(object, ..., ztol = 1e-5){
	resid <- object$resid
	nrA <- object$nrA
	tau <- object$tau
	n <- length(resid) - nrA[length(nrA)]
	Rho <- function(u,tau)sum(u*(tau-(u < 0)))
	fidelity <- Rho(resid[1:n],tau)
	penalty <- rep(NA, length(nrA)-1)
	qssedfs <- rep(NA, length(nrA)-1)
	for(i in 2:length(nrA)){
		penalty[i-1] <- sum(abs(resid[n + ((nrA[i-1]+1):nrA[i])]))
		qssedfs[i-1] <- sum(abs(resid[n + ((nrA[i-1]+1):nrA[i])]) > ztol)
		}
	penalty <- penalty/object$lambdas
	edf <- sum(abs(resid[1:n]) < ztol)
	z <- list(fidelity = fidelity, penalty = penalty,
		edf = edf, qssedfs = qssedfs, lambdas = object$lambdas)
	class(z) <- "summary.rqss"
	return(z)
	}
"logLik.rqss" <- function(object, ...){
	n <- length(object$resid) - object$nrA[length(object$nrA)]
	tau <- object$tau
	z <- summary(object)
	val <- n * (log(tau * (1-tau)) - 1 - log(z$fidelity/n))
	attr(val,"n") <- n
	attr(val,"df") <- z$edf
	class(val) <- "logLik"
	val
	}
"AIC.rqss" <- function(object, ... , k = 2){
	v <- logLik(object)
	if(k < 0) 
		k <- log(attr(v,"n"))
	val <- AIC(logLik(object), k = k)
	attr(val,"edf") <- attr(v,"df")
	val
	}
