crq <- function (formula, taus, data, subset, weights, na.action, 
	method = "FP", contrasts = NULL, ...)
{
    require(survival)
    call <- match.call()
    mf <- match.call(expand.dots = FALSE)
    m <- match(c("formula", "data", "subset", "weights", "na.action"),
        names(mf), 0)
    mf <- mf[c(1, m)]
    mf$drop.unused.levels <- TRUE
    mf[[1]] <- as.name("model.frame")
    mf <- eval.parent(mf)
    if (method == "model.frame")
        return(mf)
    mt <- attr(mf, "terms")
    X <- model.matrix(mt, mf, contrasts)
    weights <- model.weights(mf)
    Y <- model.extract(mf, "response")
    eps <- .Machine$double.eps^(2/3)
    if(!inherits(Y,"Surv"))
        stop("Response must be a survival object")
    if(method == "FP") {
	type <- attr(Y, "type")
	if(!type %in% c("right","left")) 
             stop("Only right or left censoring Surv objects are allowed")
	left <- (type == "left")
        if (any(taus < -eps) || any(taus > 1 + eps))
            stop("invalid taus:  taus should be >= 0 and <= 1")
	y <- Y[,1]
	cen <- Y[,2]
	if (length(taus) > 1) {
          coef <- matrix(0, ncol(X), length(taus))
          fitted <- resid <- matrix(0, nrow(X), length(taus))
          for (i in 1:length(taus)) {
            z <- crq.fit.FP(X, y, cen, tau = taus[i], weights, ...)
            coef[, i] <- z$coefficients
            resid[, i] <- z$residuals
            fitted[, i] <- y - z$residuals
          }
          taulabs <- paste("tau=", format(round(taus, 3)))
          dimnames(coef) <- list(dimnames(X)[[2]], taulabs)
          dimnames(resid) <- list(dimnames(X)[[1]], taulabs)
          fit <- list(coefficients = coef, residuals = resid, fitted.values = fitted)
	  fit$tau <- taus
	  class(fit) <- "crqsFP"
	}
	else {
          fit <- crq.fit.FP(X, y, cen, tau = taus, weights, ...)
	  fit$tau <- taus
	  class(fit) <- "crqFP"
	}
     }
    else if(method == "PH"){
        if(attr(Y,"type") != "right")
            stop("Only right censoring Surv objects are allowed for Peng-Huang method")
        y <- Y[,1]
        cen <-  Y[,2]
        fit <- crq.fit.PH(X, y, cen, weights,  ...)
	class(fit) <- "crqPH"
	}
    else
	stop("Method not defined for crq")

fit$terms <- mt
fit$call <- call
fit$formula <-  formula(mt)
fit$method <-  method
attr(fit, "na.message") <- attr(m, "na.message")
fit
}

crq.fit.PH <- function(x, y, cen, weights=NULL,
        grid = seq(0,1,by=min(0.01,1/(2*length(y)^.7)))) {
      p <- ncol(x)
      n <- length(y)
      m <- length(grid)
      xbar <- apply(x,2,mean)
      if(length(weights)){
                if (any(weights < 0))
                        stop("negative weights not allowed")
                contr <- attr(x, "contrasts")
                x <- x * weights
                y <- y * weights
                }
        s <- rep(0,n)
        u <- rep(1,n)
        d <- rep(1,n)
        r <- rep(1,p)
        B <- matrix(0,p,m)
        cc <- as.logical(cen)
        y1 <- y[cc]
	n1 <- length(y1)
        x1 <- x[cc,]
	z <- .Fortran("crqfnb", as.integer(n), as.integer(p), 
		a1 = as.double(t(as.matrix(x1))), c1 = as.double(-y1), n1=as.integer(n1),
		as.double(x), as.double(y),as.double(cen),B =as.double(B),
		g = as.double(grid),m = as.integer(m), as.double(r), 
		as.double(s), as.double(d), as.double(u),
        	wn = double(n1 * 9), wp = double((p + 3) * p),
        	info = integer(1), PACKAGE = "quantreg")
	B <- matrix(-z$B, p, m)
	B <- B[,1:z$m]
        qhat <- t(xbar) %*% B
        B <- rbind(grid[1:z$m],B,qhat)
        dimnames(B) <- list(c("tau",dimnames(x)[[2]],"Qhat"),NULL)
        B  <- list(sol=B)
	class(B) <- "crqPH"
	B
        }

# Function to implement (Powell) Type 1 (fixed) censored quantile regression 
# Calling routine for the BRCEN algorithm of Fitzenberger (1996, 1997) 
# (Created 12/07/98, last modified by RWK 27/11/07)


crq.fit.FP <- function(x, y, yc, tau=0.5, weights, start, left=TRUE){

 tol  <- .Machine$double.eps^(2/3)
 x <- as.matrix(x)
 y <-as.vector(y)
 n <- nrow(x)
 p <- ncol(x)
 if(missing(yc)) yc <- rep(0,n) 
 yc <- as.vector(yc)
 if(length(weights)){
   if (any(weights < 0))
       stop("negative weights not allowed")
   contr <- attr(x, "contrasts")
   x <- x * weights
   y <- y * weights
   }
 if(left) {
	x <- -x
	y <- -y
	yc <- -yc
	tau <- 1-tau
	}

 cens <- rep(0, n)
 cens[ (yc - y ) < tol ] <- 1


# Starting value of beta kludge
if(!missing(start)){ 
   if(length(start) == p){
	y <- y - x %*% start
	yc <- yc - x %*% start
	}
   else
	stop("start vector must be of length ncol(X)")
   }

optimum <- FALSE
 
qrc <- .Fortran("qrcens", 
	as.double(x),
	as.double(y),
	as.double(yc),
	as.integer(p),
	as.integer(n),
	coef = double(p),
	double((n+1)*(p+2)),
	as.integer(n + 1),
	as.integer(p + 2),
	double(n),
	double(n),
	integer(n),
	as.integer(cens),
	integer(n),
	integer(p),
	as.double(tol),
	as.logical(optimum),
	as.double(tau),
	eflag = as.integer(0),
	wflag = as.integer(0),
	PACKAGE = "quantreg")

if(qrc$eflag!=0)
	stop(switch(qrc$eflag, "Censoring indicator is incorrect ",
		"x and beta don't match "))
if(qrc$wflag!=0) 
	warning(switch(qrc$wflag, "Max iterations reached",
		"Solution may be nonunique","Tabcheck problem"))
coef <- qrc$coef
if(!missing(start))
	coef <- coef + start
residuals <- as.matrix(y - pmin(yc, x %*% coef))

return (list(coefficients= coef, residuals = residuals))
}

summary.crqFP <-
function (object, se = "boot", covariance = TRUE, ...) 
{
    mt <- terms(object)
    m <- model.frame(object)
    x <- model.matrix(mt, m, contrasts = object$contrasts)

    Y <- model.response(m)
    y <- Y[,1]
    cen <- Y[,2]
    wt <- model.weights(object$model)
    tau <- object$tau
    eps <- .Machine$double.eps^(2/3)
    coef <- coefficients(object)
    if (is.matrix(coef)) 
        coef <- coef[, 1]
    vnames <- dimnames(x)[[2]]
    resid <- object$residuals
    n <- length(resid)
    p <- length(coef)
    rdf <- n - p
    if (!is.null(wt)) {
        resid <- resid * wt
        x <- x * wt
        y <- y * wt
    }
    if (missing(se)) {
        se <- "boot"
    }
    if (se == "ker") {
        stop("ker method not yet implemented for method FP")
    }
    else if (se == "boot") {
        s <- y > x %*% coef
        B <- boot.rq(x[s, ], y[s], tau, ...)
        cov <- cov(B)
        serr <- sqrt(diag(cov))
    }
    coef <- array(coef, c(p, 4))
    dimnames(coef) <- list(vnames, c("Value", "Std. Error", "t value", 
        "Pr(>|t|)"))
    coef[, 2] <- serr
    coef[, 3] <- coef[, 1]/coef[, 2]
    coef[, 4] <- if(rdf > 0) 
        2 * (1 - pt(abs(coef[, 3]),rdf)) 
    else NA
    object <- object[c("call", "terms")]
    if (covariance == TRUE) 
        object$cov <- cov
    object$B <- B
    object$coefficients <- coef
    object$rdf <- rdf
    object$tau <- tau
    class(object) <- "summary.crqFP"
    object
}
"print.crqFP" <-
function(x, ...)
{
        if(!is.null(cl <- x$call)) {
                cat("Call:\n")
                dput(cl)
        }
        coef <- coef(x)
        cat("\nCoefficients:\n")
        print(coef, ...)
        rank <- x$rank
        nobs <- length(residuals(x))
        if(is.matrix(coef))
                p <- dim(coef)[1]
        else p <- length(coef)
        rdf <- nobs - p
        cat("\nDegrees of freedom:", nobs, "total;", rdf, "residual\n")
        if(!is.null(attr(x, "na.message")))
                cat(attr(x, "na.message"), "\n")
        invisible(x)
}

print.summary.crqsFP <- function(x, ...)
    lapply(x,print.summary.crqPH)
print.summary.crqFP <- function (x, digits = max(5, .Options$digits - 2), ...) {
    coef <- x$coefficients
    tau <- x$tau
    cat("\ntau: ")
    print(format(round(tau, digits = digits)), quote = FALSE, ...)
    cat("\nCoefficients:\n")
    print(format(round(coef, digits = digits)), quote = FALSE, ...)
    invisible(x)
}


summary.crqPH <-
function (object, taus = 1:4/5, alpha = .05, se = "boot", ...) 
{
    mt <- terms(object)
    m <- model.frame(object)
    Y <- model.response(m)
    y <- Y[,1]
    cen  <- Y[,2]
    x <- model.matrix(mt, m, contrasts = object$contrasts)
    coef <- coef(object,taus)
    coef <- coef[-nrow(coef),] #delete Qbar row
    coef <- coef[,apply(coef,2,function(x) any(!is.na(x)))] # Delete NA columns if any
    taus <- taus[1:ncol(coef)]
    meth <- object$method
    B <- boot.crqPH(x, y, cen, taus, ...)
    sqmn <- sqrt(B$mboot/B$n)
    fact <-   qnorm(1 - alpha/2)/qnorm(.75)
    B <- apply(B$A, 1:2, quantile, probs = 1:3/4, na.rm = TRUE)
    D <- .5 * fact *(B[3,,]-B[1,,]) * sqmn
    L <- coef - D
    U <- coef + D
    S <- (U - L)/(2 * qnorm(.75))
    T <- coef/S
    P <- 2 * (1 - pnorm(abs(T)))
    G <- list()
    cnames <- c("Value","Lower Bd","Upper Bd","Std Error","T Value","Pr(>|t|)")
    for(i in 1:length(taus)){
	tab <- cbind(coef[,i],L[,i],U[,i],S[,i],T[,i],P[,i])
	dimnames(tab)[[2]] <- cnames
	G[[i]] <- list(tau = taus[i], coefficients = tab)
	}
    class(G) <- "summary.crqsPH"
    G
   }
print.crqPH <- function(x, ...)
    print(coef(x, ...), ...)
print.summary.crqsPH <- function(x, ...)
    lapply(x,print.summary.crqPH)
print.summary.crqPH <- function (x, digits = max(5, .Options$digits - 2), ...) {
    coef <- x$coefficients
    tau <- x$tau
    cat("\ntau: ")
    print(format(round(tau, digits = digits)), quote = FALSE, ...)
    cat("\nCoefficients:\n")
    print(format(round(coef, digits = digits)), quote = FALSE, ...)
    invisible(x)

}

plot.summary.crqsPH <-
function (x, nrow = 3, ncol = 3, CoxPHit = NULL, ...) {
    taus <- function(x) x$tau
    xx <- unlist(lapply(x, taus))
    coef <- lapply(x, coefficients)
    p <- nrow(coef[[1]])
    k <- ncol(coef[[1]])
    if(k != 6) stop("summary.crqs object has wrong column dimension")
    m <- length(xx)
    blab <- dimnames(coef[[1]])[[1]]
    a <- array(unlist(coef), c(p, k, m))
    if(length(CoxPHit))
	CoxQTE <- QTECox(CoxPHit)
    par(mfrow = c(nrow, ncol))
    for (i in 2:p) {
            b  <- a[i, 1, ]
            bl <- a[i, 2, ]
            bu <- a[i, 3, ]
        plot(rep(xx, 2), c(bl, bu), xlab = "", ylab = "", type = "n")
        title(paste(blab[i]), cex = 0.75)
        polygon(c(xx, rev(xx)), c(bl, rev(bu)), col = "LightSkyBlue")
        points(xx, b, cex = 0.5, pch = "o", col = "blue")
        lines(xx, b, col = "blue")
        abline(h = 0)
        if(length(CoxPHit)) {
	    lines(CoxQTE$taus,CoxQTE$QTE[,i-1],col="red")
            }
    }
}
QTECox <- function(x, smooth = TRUE){
# compute quantile treatment effect for a Cox PH fit
g <- survfit(x)
if(smooth)
        g <- supsmu(1-g$surv,g$time)
taus <- (g$x[-1] + g$x[-length(g$x)])/2
Qhat <- (g$y[-1] + g$y[-length(g$y)])/2

dQ <- diff(Qhat)/diff(taus)
taus <- taus[-1]; Qhat <- Qhat[-1]
QTE <- outer(dQ * (1-taus) * log(1-taus) / Qhat, coef(x))
list(QTE = QTE , taus = taus)
}


coef.crqPH <- function(object, taus = 1:4/5, ...)
        {
        # Extract coefficients from the crqPH solution array 
        
        if(min(taus) < 0 || max(taus) > 1) stop("taus out of range [0,1]")
        taus <- sort(taus)
        S <- object$sol
        r <- S[1, ]
        r <- c(r[1],r)
        r <- (r[-1]+r[-length(r)])/2
        B <- S[-1,]
        J <- length(r)
        ts <- taus[taus < max(r)]
        bin <- cut(ts,r,label=FALSE)
        wgt <- (ts - r[bin])/(r[bin + 1] - r[bin])
        coef <- wgt * B[,bin] + (1-wgt) * B[,bin - 1]
        nna <- length(taus) - length(ts)
        if(nna > 0)
                coef <- cbind(coef, matrix(NA,nrow(coef),nna))
        taulabs <- paste("tau=", format(round(taus, 3)))
        dimnames(coef)[[2]] <- taulabs
        coef
        }

FSurv <- function (y,  yc, type = c("left", "right")) 
{
    nn <- length(y)
    if(length(yc) != nn)
        stop("Event times and censoring times of different length")
    type <- match.arg(type)
    if(type == "right" && any(y > yc))
        stop("Event times can not exceed ctimes for right censoring")
    if(type == "left" && any(y < yc))
        stop("Event times can not be less than ctimes for left censoring")
    if(!(type == "left" || type == "right"))
        stop("Invalid type for method FP")
    ss <- cbind(y, yc)
    dimnames(ss) <- list(NULL, c("time", "ctime"))
    attr(ss, "type") <- type
    class(ss) <- "Surv"
    ss
}

