\name{crq}
\alias{crq}
\alias{crq.fit.PH}
\alias{crq.fit.FP}
\alias{print.crqFP}
\alias{print.crqPH}
\alias{coef.crqPH}
\title{Function to fit a censored quantile regression model} 
\description{Fits a conditional quantile regression model for censored data. There
are (currently) two distinct methods:  the first is the fixed censoring method
of Powell (1986) as implemented by Fitzenberger (1996), the second is the random
censoring method of Peng and Huang (2008). A third method is available in the
separate package \link{crq}.}
\usage{
crq(formula, taus, data, subset, weights, na.action, method = "FP", contrasts = NULL, ...)
crq.fit.FP(x, y, yc, tau=0.5, weights, start, left=TRUE)
crq.fit.PH(x, y, cen, weights=NULL, grid = seq(0,1,by=min(0.01,1/(2*length(y)^.7)))) 
\method{print}{crqFP}(x, ...)
\method{print}{crqPH}(x, ...)
\method{coef}{crqPH}(object,taus = 1:4/5,...)
}
\arguments{
  \item{formula}{A formula object, with the response on the left of the `~'
        operator, and the terms on the right.  The response must be a
        survival object as returned by the \code{Surv} function. For
	the Fitzenberger-Powell (FP) method, the Surv object should
	have arguments (event time, censoring time,type), where "type"
	can take values either "left" or "right" while for the Peng
	and Huang (PH) method Surv should have (event time, censoring
	indicator).  The default (for historical reasons) for type
	in the FP case is "left".}
  \item{y}{The event time.}
  \item{grid}{A vector of taus on which the quantile process should be evaluated.}
  \item{x}{An object of class \code{crqFP} or \code{crqPH}.}
  \item{object}{An object of class \code{crqFP} or \code{crqPH}.}
  \item{yc}{The censoring times for the "FP" method.}
  \item{cen}{The censoring indicator for the "PH" method.}
  \item{start}{The starting value for the coefs for the "FP" method.}
  \item{left}{An logical indicator for left censoring for the "FP" method.}
  \item{taus}{The quantile(s) at which the model is to be estimated.}
  \item{tau}{The quantile at which the model is to be estimated.}
  \item{data}{A data.frame in which to interpret the variables named in the
          `formula',  in the `subset', and the `weights' argument.}
  \item{subset}{an optional vector specifying a subset of observations to be
          used in the fitting process.}
  \item{weights}{vector of observation weights; if supplied, the algorithm
          fits to minimize the sum of the weights multiplied into the
          absolute residuals. The length of weights vector must be the same as
          the number of observations.  The weights must be nonnegative
          and it is strongly recommended that they be strictly
          positive, since zero weights are ambiguous.} 
  \item{na.action}{a function to filter missing data.  This is applied to the
          model.frame after any subset argument has been used.  The
          default (with 'na.fail') is to create an error if any missing
          values are   found.  A possible alternative is 'na.omit',
          which  deletes observations that contain one or more missing
          values. } 
  \item{method}{The method used for fitting.  There are currently
	two options: method "FP" computes the Powell estimator using
	the algorithm of Fitzenberger (1996), method "PH" computes the
	Peng and Huang (2008) estimator, on a specified grid of tau values in (0,1). 
	The other method is "pivot" which fits the process on a grid
	determined by the data; this method is analogous to fitting the
	whole rq process with \code{rq}.  If method is "PH" then the method
	of Peng and Huang (2007) is used, in this case the variable "grid"
	can be passed to specify the vector of quantiles at which the solution
	is desired.}
  \item{contrasts}{a list giving contrasts for some or all of the factors 
          default = 'NULL' appearing in the model formula.  The
          elements of the list should have the same name as the
          variable  and should be either a contrast matrix
          (specifically, any full-rank  matrix with as many rows as
          there are levels in the factor),  or else a function to
          compute such a matrix given the number of levels.} 
  \item{...}{additional arguments for the fitting routine, for method "FP"
	it may be useful to pass starting values of the regression parameter
	via "start = initialbeta", while for method "PH" one may wish to
	specify an alternative to the default grid for evaluating the fit.}
}
\details{The Fitzenberger algorithm uses a variant of the Barrodale and Roberts
	simplex method, but since the objective function is no longer convex
	there can be no guarantee that it produces a global minimum estimate.
	In small problems exhaustive search over solutions defined by p-element
	subsets of the n observations can be used, but this quickly becomes
	impractical for large p and n.   The Peng and Huang estimator can be
	viewed as a generalization of the Nelson Aalen estimator of the cumulative
	hazard function,  and can be formulated as a variant of the conventional
	quantile regression dual problem.  See Koenker (2008) for further details.}
\value{An object of class \code{crqFP} or \code{crqPH}.}
}
\references{
Powell, J. (1986) ``Censored Regression Quantiles,'' \emph{J.
Econometrics}, 32, 143--155.

Fitzenberger, B.  (1996): ``A Guide to Censored Quantile
Regressions,'' in \emph{Handbook of Statistics}, ed. by C.~Rao,   and
G.~Maddala. North-Holland: New York.

Peng, L and Y Huang, (2008) Survival Analysis with Quantile Regression Models,
\emph{J. Am. Stat. Assoc.}, forthcoming.
}


\author{Bernd Fitzenberger and Roger Koenker}

\seealso{ \code{\link{crq.fit.FP}}, \code{\link{crq.fit.PH}}, 
\code{\link{summary.crqFP}}, \code{\link{summary.crqPH}}
}

\examples{
# An artificial Powell example
set.seed(2345)
x <- sqrt(rnorm(100)^2)
y <-  -0.5 + x +(.25 + .25*x)*rnorm(100)
plot(x,y, type="n")
s <- (y > 0)
points(x[s],y[s],cex=.9,pch=16)
points(x[!s],y[!s],cex=.9,pch=1)
yLatent <- y
y <- pmax(0,y)
yc <- rep(0,100)
for(tau in (1:4)/5){
        f <- crq(Surv(y,yc) ~ x, tau = tau, method = "FP")
        xs <- sort(x)
        lines(xs,pmax(0,cbind(1,xs)\%*\%f$coef),col="red")
        abline(rq(y ~ x, tau = tau), col="blue")
        abline(rq(yLatent ~ x, tau = tau), col="green")
        }
legend(.15,2.5,c("Naive QR","Censored QR","Omniscient QR"),
        lty=rep(1,3),col=c("blue","red","green"))

data(uis)
#estimate the Peng and Huang model using log(TIME) AFT specification
fit <- crq(Surv(log(TIME), CENSOR) ~  ND1 + ND2 + IV3 +
               TREAT + FRAC + RACE + AGE * SITE, method = "PH", data = uis)
Sfit <- summary(fit,1:19/20)
PHit <- coxph(Surv(TIME, CENSOR) ~  ND1 + ND2 + IV3 +
               TREAT + FRAC + RACE + AGE * SITE, data = uis)
plot(Sfit, CoxPHit = PHit)
}
\keyword{survival}
\keyword{regression}
