"combn" <- 
function(x, m, fun = NULL, simplify = T, ...)
{
	#       DATE WRITTEN: 14 April 1994          LAST REVISED:  10 July 1995
	#       AUTHOR:  Scott Chasalow
	#
	#       DESCRIPTION:
	#             Generate all combinations of the elements of x taken m at a time. 
	#             If x is a positive integer,  returns all combinations
	#             of the elements of seq(x) taken m at a time.
	#             If argument "fun" is not null,  applies a function given
	#             by the argument to each point.  If simplify is FALSE,  returns 
	#             a list; else returns a vector or an array.  "..." are passed 
	#             unchanged to function given by argument fun,  if any.
	#       REFERENCE:
	#             Nijenhuis, A. and Wilf, H.S. (1978) Combinatorial Algorithms for 
	#             Computers and Calculators.  NY:  Academic Press.
	#       EXAMPLES:
	#             > combn(letters[1:4], 2)
	#             > combn(10, 5, min)  # minimum value in each combination
	#             Different way of encoding points:
	#             > combn(c(1,1,1,1,2,2,2,3,3,4), 3, tabulate, nbins = 4)
	#             Compute support points and (scaled) probabilities for a
	#             Multivariate-Hypergeometric(n = 3, N = c(4,3,2,1)) p.f.:
	#             > table.mat(t(combn(c(1,1,1,1,2,2,2,3,3,4), 3, tabulate,nbins=4)))
	#
	if(length(m) > 1) {
		warning(paste("Argument m has", length(m), 
			"elements: only the first used"))
		m <- m[1]
	}
	if(m < 0)
		stop("m < 0")
	if(m == 0)
		return(if(simplify) vector(mode(x), 0) else list())
	if(is.numeric(x) && length(x) == 1 && x > 0 && trunc(x) == x)
		x <- seq(x)
	n <- length(x)
	if(n < m)
		stop("n < m")
	e <- 0
	h <- m
	a <- 1:m
	nofun <- is.null(fun)
	count <- nCm(n, m, 0.10000000000000002)
	out <- vector("list", count)
	out[[1]] <- if(nofun) x[a] else fun(x[a], ...)
	if(simplify) {
		dim.use <- NULL
		if(nofun) {
			if(count > 1)
				dim.use <- c(m, count)
		}
		else {
			out1 <- out[[1]]
			d <- dim(out1)
			if(count > 1) {
				if(length(d) > 1)
					dim.use <- c(d, count)
				else if(length(out1) > 1)
					dim.use <- c(length(out1), count)
			}
			else if(length(d) > 1)
				dim.use <- d
		}
	}
	i <- 2
	nmmp1 <- n - m + 1
	mp1 <- m + 1
	while(a[1] != nmmp1) {
		if(e < n - h) {
			h <- 1
			e <- a[m]
			j <- 1
		}
		else {
			h <- h + 1
			e <- a[mp1 - h]
			j <- 1:h
		}
		a[m - h + j] <- e + j
		out[[i]] <- if(nofun) x[a] else fun(x[a], ...)
		i <- i + 1
	}
	if(simplify) {
		if(is.null(dim.use))
			out <- unlist(out)
		else out <- array(unlist(out), dim.use)
	}
	out
}

"combn2" <- 
function(x, n)
{
	#   DATE WRITTEN:  14 April 1994           LAST REVISED:  14 April 1994
	#   AUTHOR:  Scott D. Chasalow
	#
	#   DESCRIPTION:
	#         Generate all combinations of the elements of x taken two at a time. 
	#         If x is missing,  generate all combinations of 1:n taken two
	#         at a time (that is,  the indices of x that would give all 
	#         combinations of the elements of x if x with length n had been given).
	#         Exactly one of arguments "x" and "n" should be given.
	#
	if(!missing(x)) {
		n <- length(x)
		if(!missing(n))
			warning(paste("Only one of arguments x and n allowed;",
				"argument n was ignored"))
	}
	else if(missing(n))
		stop("Arguments \"x\" and \"n\" both missing")
	if(length(n) > 1) {
		warning(paste("Argument n has", length(n), 
			"elements: only the first used"))
		n <- n[1]
	}
	if(n == 0)
		return(NULL)
	rmat <- array(seq(length = n), c(n, n))
	# row(matrix(0,n,n))
	cmat <- t(rmat)
	# col(matrix(0,n,n))
	lower.t <- rmat > cmat
	# lower.tri(matrix(0,n,n))
	i1 <- cmat[lower.t]
	i2 <- rmat[lower.t]
	if(missing(x))
		cbind(i1, i2)
	else cbind(x[i1], x[i2])
}

"deld" <- 
function(x, y, xadd = NULL, yadd = NULL, dp.str = NULL, rw = NULL, eps = 
	1.0000000000000001e-09, frac = 0.0001, sort = T, plotit = F, digits = 6,
	...)
{
	# Function deldir
	#
	#   Copyright (C) 1996 by T. Rolf Turner
	#
	#   Permission to use, copy, modify, and distribute this software and
	#   its documentation for any purpose and without fee is hereby
	#   granted, provided that the above copyright notice appear in all
	#   copies and that both that copyright notice and this permission
	#   notice appear in supporting documentation.
	#
	# ORIGINALLY PROGRAMMED BY: Rolf Turner in 1987/88, while with the
	# Division of Mathematics and Statistics, CSIRO, Sydney, Australia.
	# Re-programmed by Rolf Turner to adapt the implementation from a
	# stand-alone Fortran program to an S function, while visiting the
	# University of Western Australia, May 1995.  Further revised
	# December 1996.
	# 
	# MODIFIED BY; Roger Koenker May 1999 to accomodate adding non-basic
	# points which are simply associated with their covering triangle
	# with barycentric coordinates  provided, and to compute pseudo-obs
	# for the penalty term of the triogram problem.
	#
	# Function to compute the Delaunay Triangulation (and hence the
	# Dirichlet Tesselation) of a planar point set according to the
	# second (iterative) algorithm of Lee and Schacter, International
	# Journal of Computer and Information Sciences, Vol. 9, No. 3, 1980, 
	# pages 219 to 242.
	# The triangulation is made to be with respect to the whole plane by
	# `suspending' it from `ideal' points
	# (-R,-R), (R,-R) (R,R), and (-R,R), where R --> infinity.
	# It is also enclosed in a finite rectangle (whose boundaries truncate any
	# infinite Dirichlet tiles) with corners (xmin,ymin) etc.  This rectangle
	# is referred to elsewhere as `the' rectangular window.
	# If first argument is a list, extract components x and y.
	# If a data window is specified, get its corner coordinates
	# and truncate the data by this window.
	if(is.list(x)) {
		if(all(!is.na(match(c("x", "y"), names(x))))) {
			y <- x$y
			x <- x$x
		}
		else {
			cat("Error: called with list lacking both x and y elements\n"
				)
			return()
		}
	}
	n <- length(x)
	if(n != length(y))
		stop("data lengths do not match")
	if(!is.null(rw)) {
		xmin <- rw[1]
		xmax <- rw[2]
		ymin <- rw[3]
		ymax <- rw[4]
		drop <- (1:n)[x < xmin | x > xmax | y < ymin | y > ymax]
		if(length(drop) > 0) {
			x <- x[ - drop]
			y <- y[ - drop]
			n <- length(x)
		}
	}
	else {
		xmin <- min(x)
		xmax <- max(x)
		ymin <- min(y)
		ymax <- max(y)
		xdff <- xmax - xmin
		ydff <- ymax - ymin
		xmin <- xmin - 0.10000000000000001 * xdff
		xmax <- xmax + 0.10000000000000001 * xdff
		ymin <- ymin - 0.10000000000000001 * ydff
		ymax <- ymax + 0.10000000000000001 * ydff
		rw <- c(xmin, xmax, ymin, ymax)
	}
	# Add the dummy points.
	# Eliminate duplicate points:
	if(!is.loaded(symbol.For("acchk"))) {
		ldlib <- unix("echo $DYN_LOAD_LIB")
		if(ldlib == "")
			ldlib <- "."
		ldfile <- paste(ldlib, "/deldirld.o", sep = "")
		dyn.load(ldfile)
	}
	if(!is.null(dp.str)) {
		dpts <- dumpts(x, y, dp.str, rw)
		x <- dpts$x
		y <- dpts$y
	}
	iii <- !ind.dup(x, y, rw, frac)
	n <- sum(iii[1:n])
	ndm <- sum(iii[ - (1:n)])
	x <- x[iii]
	y <- y[iii]
	# Make space for the total number of points (real and dummy) as
	# well as 4 ideal points and 4 extra corner points which get used
	# (only) by subroutines dirseg and dirout in the ``output'' process
	# (returning a description of the triangulation after it has been
	# calculated).
	npd <- n + ndm
	# ntot doesn't include the 4 extra corners.
	ntot <- npd + 4
	x <- c(rep(0, 4), x, rep(0, 4))
	# Set up fixed dimensioning constants.
	y <- c(rep(0, 4), y, rep(0, 4))
	ntdel <- 4 * npd
	ntdir <- 3 * npd
	# Set up dimensioning constants which might need to be increased:
	madj <- max(20, ceiling(3 * sqrt(ntot)))
	tadj <- (madj + 1) * (ntot + 4)
	ndel <- (madj * (madj + 1))/2
	tdel <- 12 * ndel
	ndir <- ndel
	# Call the master subroutine to do the work:
	tdir <- 8 * ndir
	repeat {
		# Check for errors:
		tmp <- .Fortran("mast",
			x = as.double(x),
			y = as.double(y),
			sort = as.logical(sort),
			rw = as.double(rw),
			npd = as.integer(npd),
			ntot = as.integer(ntot),
			nadj = integer(tadj),
			madj = as.integer(madj),
			ind = integer(npd),
			tx = double(npd),
			ty = double(npd),
			ilist = integer(npd),
			eps = as.double(eps),
			delsgs = double(tdel),
			ndel = as.integer(ndel),
			delsum = double(ntdel),
			dirsgs = double(tdir),
			ndir = as.integer(ndir),
			dirsum = double(ntdir),
			nerror = integer(1))
		nerror <- tmp$nerror
		if(nerror < 0)
			break
		else {
			if(nerror == 4) {
				cat("nerror =", nerror, "\n")
				cat("Increasing madj and trying again.\n")
				madj <- ceiling(1.2 * madj)
				tadj <- (madj + 1) * (ntot + 4)
				ndel <- max(ndel, (madj * (madj + 1))/2)
				tdel <- 12 * ndel
				ndir <- ndel
				tdir <- 8 * ndir
			}
			else if(nerror == 14 | nerror == 15) {
				cat("nerror =", nerror, "\n")
				cat("Increasing ndel and ndir and trying again.\n"
					)
				ndel <- ceiling(1.2 * ndel)
				tdel <- 12 * ndel
				ndir <- ndel
				tdir <- 8 * ndir
			}
			else {
				cat("nerror =", nerror, "\n")
				return(invisible())
			}
		}
	}
	# Collect the results for return:
	ndel <- tmp$ndel
	delsgs <- round(t(as.matrix(matrix(tmp$delsgs, nrow = 12)[, 1:ndel])),
		digits)
	ind <- tmp$ind
	npd <- tmp$npd
	nadj <- tmp$nadj
	madj <- tmp$madj
	x <- tmp$x
	y <- tmp$y
	ntot <- tmp$ntot
	eps <- tmp$eps
	delsum <- matrix(tmp$delsum, ncol = 4)
	del.area <- sum(delsum[, 4])
	delsum <- round(cbind(delsum, delsum[, 4]/del.area), digits)
	del.area <- round(del.area, digits)
	ndir <- tmp$ndir
	dirsgs <- round(t(as.matrix(matrix(tmp$dirsgs, nrow = 8)[, 1:ndir])),
		digits)
	bpts <- dirsgs[, 7:8]
	mode(bpts) <- "logical"
	dirsgs <- data.frame(dirsgs[,  - (7:8)], bpts)
	dirsum <- matrix(tmp$dirsum, ncol = 3)
	dir.area <- sum(dirsum[, 3])
	dirsum <- round(cbind(dirsum, dirsum[, 3]/dir.area), digits)
	dir.area <- round(dir.area, digits)
	allsum <- cbind(delsum, dirsum)
	rw <- round(rw, digits)
	dimnames(delsgs) <- list(NULL, c("x1", "y1", "x2", "y2", "ind1", "ind2",
		"ind3", "ind4", "h1", "h2", "h3", "h4"))
	names(dirsgs) <- c("x1", "y1", "x2", "y2", "ind1", "ind2", "bp1", "bp2"
		)
	dimnames(allsum) <- list(NULL, c("x", "y", "n.tri", "del.area", 
		"del.wts", "n.tside", "nbpt", "dir.area", "dir.wts"))
	# Aw' done!!!
	rslt <- list(delsgs = delsgs, ind = ind, nadj = nadj, madj = madj,
		x = x, y = y, npd = npd, ntot = ntot, eps = eps, dirsgs = 
		dirsgs, summary = allsum, n.data = n, n.dum = ndm, del.area = 
		del.area, dir.area = dir.area, rw = rw)
	class(rslt) <- "deldir"
	if(plotit) {
		plot(rslt, ...)
	}
	if(plotit)
		invisible(rslt)
	else rslt
}

"deldir" <- 
function(x, y, xadd = NULL, yadd = NULL, dp.str = NULL, rw = NULL, eps = 
	1.0000000000000001e-09, frac = 0.0001, sort = T, plotit = F, digits = 6,
	...)
{
	# Function deldir
	#
	#   Copyright (C) 1996 by T. Rolf Turner
	#
	#   Permission to use, copy, modify, and distribute this software and
	#   its documentation for any purpose and without fee is hereby
	#   granted, provided that the above copyright notice appear in all
	#   copies and that both that copyright notice and this permission
	#   notice appear in supporting documentation.
	#
	# ORIGINALLY PROGRAMMED BY: Rolf Turner in 1987/88, while with the
	# Division of Mathematics and Statistics, CSIRO, Sydney, Australia.
	# Re-programmed by Rolf Turner to adapt the implementation from a
	# stand-alone Fortran program to an S function, while visiting the
	# University of Western Australia, May 1995.  Further revised
	# December 1996.
	# 
	# MODIFIED BY; Roger Koenker May 1999 to accomodate adding non-basic
	# points which are simply associated with their covering triangle
	# with barycentric coordinates  provided, and to compute pseudo-obs
	# for the penalty term of the triogram problem.
	#
	# Function to compute the Delaunay Triangulation (and hence the
	# Dirichlet Tesselation) of a planar point set according to the
	# second (iterative) algorithm of Lee and Schacter, International
	# Journal of Computer and Information Sciences, Vol. 9, No. 3, 1980, 
	# pages 219 to 242.
	# The triangulation is made to be with respect to the whole plane by
	# `suspending' it from `ideal' points
	# (-R,-R), (R,-R) (R,R), and (-R,R), where R --> infinity.
	# It is also enclosed in a finite rectangle (whose boundaries truncate any
	# infinite Dirichlet tiles) with corners (xmin,ymin) etc.  This rectangle
	# is referred to elsewhere as `the' rectangular window.
	# If first argument is a list, extract components x and y.
	# Check dimensions of xadd,yadd
	if(is.list(x)) {
		if(all(!is.na(match(c("x", "y"), names(x))))) {
			y <- x$y
			x <- x$x
		}
		else {
			cat("Error: called with list lacking both x and y elements\n"
				)
			return()
		}
	}
	nadd <- length(xadd)
	if(nadd != length(yadd))
		stop("(xadd,yadd) data lengths do not match")
	if(nadd == 0) {
		nadd <- 1
		xadd[1] <- 0
		yadd[1] <- 0
	}
	xx <- x
	# If a data window is specified, get its corner coordinates
	yy <- y
	# and truncate the data by this window.
	n <- length(x)
	if(n != length(y))
		stop("data lengths do not match")
	if(!is.null(rw)) {
		xmin <- rw[1]
		xmax <- rw[2]
		ymin <- rw[3]
		ymax <- rw[4]
		drop <- (1:n)[x < xmin | x > xmax | y < ymin | y > ymax]
		if(length(drop) > 0) {
			x <- x[ - drop]
			y <- y[ - drop]
			n <- length(x)
		}
	}
	else {
		xmin <- min(x)
		xmax <- max(x)
		ymin <- min(y)
		ymax <- max(y)
		xdff <- xmax - xmin
		ydff <- ymax - ymin
		xmin <- xmin - 0.10000000000000001 * xdff
		xmax <- xmax + 0.10000000000000001 * xdff
		ymin <- ymin - 0.10000000000000001 * ydff
		ymax <- ymax + 0.10000000000000001 * ydff
		rw <- c(xmin, xmax, ymin, ymax)
	}
	# Add the dummy points.
	# Eliminate duplicate points:
	if(!is.loaded(symbol.For("acchk"))) {
		ldlib <- unix("echo $DYN_LOAD_LIB")
		if(ldlib == "")
			ldlib <- "."
		ldfile <- paste(ldlib, "/deldirld.o", sep = "")
		dyn.load(ldfile)
	}
	if(!is.null(dp.str)) {
		dpts <- dumpts(x, y, dp.str, rw)
		x <- dpts$x
		y <- dpts$y
	}
	iii <- !ind.dup(x, y, rw, frac)
	n <- sum(iii[1:n])
	ndm <- sum(iii[ - (1:n)])
	x <- x[iii]
	y <- y[iii]
	# Make space for the total number of points (real and dummy) as
	# well as 4 ideal points and 4 extra corner points which get used
	# (only) by subroutines dirseg and dirout in the ``output'' process
	# (returning a description of the triangulation after it has been
	# calculated).
	npd <- n + ndm
	# ntot doesn't include the 4 extra corners.
	ntot <- npd + 4
	x <- c(rep(0, 4), x, rep(0, 4))
	# Set up fixed dimensioning constants.
	y <- c(rep(0, 4), y, rep(0, 4))
	ntdel <- 4 * npd
	ntdir <- 3 * npd
	# Set up dimensioning constants which might need to be increased:
	madj <- max(20, ceiling(3 * sqrt(ntot)))
	tadj <- (madj + 1) * (ntot + 4)
	ndel <- (madj * (madj + 1))/2
	tdel <- 12 * ndel
	ndir <- ndel
	# Call the master subroutine to do the work:
	tdir <- 8 * ndir
	repeat {
		# Check for errors:
		tmp <- .Fortran("master",
			x = as.double(x),
			y = as.double(y),
			xadd = as.double(xadd),
			yadd = as.double(yadd),
			nadd = as.integer(nadd),
			sort = as.logical(sort),
			rw = as.double(rw),
			npd = as.integer(npd),
			ntot = as.integer(ntot),
			nadj = integer(tadj),
			madj = as.integer(madj),
			ind = integer(npd),
			tx = double(npd),
			ty = double(npd),
			ilist = integer(npd),
			eps = as.double(eps),
			delsgs = double(tdel),
			ndel = as.integer(ndel),
			delsum = double(ntdel),
			dirsgs = double(tdir),
			ndir = as.integer(ndir),
			dirsum = double(ntdir),
			triadd = integer(3 * nadd),
			tribar = double(3 * nadd),
			nerror = integer(1))
		nerror <- tmp$nerror
		if(nerror < 0)
			break
		else {
			if(nerror == 4) {
				cat("nerror =", nerror, "\n")
				cat("Increasing madj and trying again.\n")
				madj <- ceiling(1.2 * madj)
				tadj <- (madj + 1) * (ntot + 4)
				ndel <- max(ndel, (madj * (madj + 1))/2)
				tdel <- 12 * ndel
				ndir <- ndel
				tdir <- 8 * ndir
			}
			else if(nerror == 14 | nerror == 15) {
				cat("nerror =", nerror, "\n")
				cat("Increasing ndel and ndir and trying again.\n"
					)
				ndel <- ceiling(1.2 * ndel)
				tdel <- 12 * ndel
				ndir <- ndel
				tdir <- 8 * ndir
			}
			else {
				cat("nerror =", nerror, "\n")
				return(invisible())
			}
		}
	}
	# Collect the results for return:
	# Invert the permutation tmp$ind used to bin data: see Knuth, AoCP, v3 p13.
	indinv <- (1:tmp$npd)[order(tmp$ind)]
	browser()
	if(nadd > 1) {
		triadd <- indinv[c(tmp$triadd)]
		tribar <- tmp$tribar
		vx <- xx[triadd]
		vy <- yy[triadd]
		triadd <- t(matrix(triadd, 3))
		tribar <- t(matrix(tribar, 3))
	}
	else {
		triadd <- NULL
		tribar <- NULL
	}
	ndel <- tmp$ndel
	delsgs <- round(t(as.matrix(matrix(tmp$delsgs, nrow = 12)[, 1:ndel])),
		digits)
	delsum <- matrix(tmp$delsum, ncol = 4)
	del.area <- sum(delsum[, 4])
	delsum <- round(cbind(delsum, delsum[, 4]/del.area), digits)
	del.area <- round(del.area, digits)
	ndir <- tmp$ndir
	dirsgs <- round(t(as.matrix(matrix(tmp$dirsgs, nrow = 8)[, 1:ndir])),
		digits)
	bpts <- dirsgs[, 7:8]
	mode(bpts) <- "logical"
	dirsgs <- data.frame(dirsgs[,  - (7:8)], bpts)
	dirsum <- matrix(tmp$dirsum, ncol = 3)
	dir.area <- sum(dirsum[, 3])
	dirsum <- round(cbind(dirsum, dirsum[, 3]/dir.area), digits)
	dir.area <- round(dir.area, digits)
	allsum <- cbind(delsum, dirsum)
	rw <- round(rw, digits)
	dimnames(delsgs) <- list(NULL, c("x1", "y1", "x2", "y2", "ind1", "ind2",
		"ind3", "ind4", "h1", "h2", "h3", "h4"))
	names(dirsgs) <- c("x1", "y1", "x2", "y2", "ind1", "ind2", "bp1", "bp2"
		)
	dimnames(allsum) <- list(NULL, c("x", "y", "n.tri", "del.area", 
		"del.wts", "n.tside", "nbpt", "dir.area", "dir.wts"))
	# Aw' done!!!
	rslt <- list(delsgs = delsgs, dirsgs = dirsgs, summary = allsum, n.data
		 = n, n.dum = ndm, del.area = del.area, dir.area = dir.area,
		rw = rw, triadd = triadd, tribar = tribar)
	class(rslt) <- "deldir"
	if(plotit) {
		plot(rslt, ...)
		points(xadd, yadd)
		points(vx, vy, pch = "O")
	}
	if(plotit)
		invisible(rslt)
	else rslt
}

"dmnom" <- 
function(x, size = sum(x), prob = stop("no prob arg"))
{
	#       DATE WRITTEN: 22 May 1995           LAST REVISED:  22 May 1995
	#       AUTHOR:  Scott Chasalow
	#
	p <- max(length(x), length(prob))
	x <- rep(x, length = p)
	prob <- rep(prob, length = p)
	prob <- prob/sum(prob)
	if(sum(x) != size)
		0
	else exp(logfact(size) + sum(x * log(prob) - logfact(x)))
}

"dumpts" <- 
function(x, y, dp.str, rw)
{
	#
	# Function dumpts to append a sequence of dummy points to the
	# data points.
	#
	ndm <- 0
	xd <- NULL
	yd <- NULL
	xmin <- rw[1]
	xmax <- rw[2]
	ymin <- rw[3]
	# Points on radii of circles emanating from data points:
	ymax <- rw[4]
	# Ad hoc points passed over as part of dp.str:
	# Delete dummy points outside the rectangular window.
	if(!is.null(dp.str$nrad)) {
		# Number of radii from each data point.
		nrad <- dp.str$nrad
		# Number of dummy points per radius.
		nper <- dp.str$nper
		# Length of each radius = fctr * mean
		fctr <- dp.str$fctr
		# interpoint distance.
		lrad <- (fctr * mipd(x, y))/nper
		theta <- (2 * pi * (1:nrad))/nrad
		cs <- cos(theta)
		sn <- sin(theta)
		xt <- c(lrad * (1:nper) %o% cs)
		yt <- c(lrad * (1:nper) %o% sn)
		xd <- c(outer(x, xt, "+"))
		yd <- c(outer(y, yt, "+"))
	}
	if(!is.null(dp.str$x)) {
		xd <- c(xd, dp.str$x)
		yd <- c(yd, dp.str$y)
	}
	ndm <- length(xd)
	if(ndm > 0) {
		drop <- (1:ndm)[xd < xmin | xd > xmax | yd < ymin | yd > ymax]
		if(length(drop) > 0) {
			xd <- xd[ - drop]
			yd <- yd[ - drop]
		}
	}
	if(!is.null(dp.str$ndx)) {
		ndx <- dp.str$ndx
		ndy <- dp.str$ndy
		xt <- if(ndx > 1) seq(xmin, xmax, length = ndx) else 0.5 * (
				xmin + xmax)
		yt <- if(ndy > 1) seq(ymin, ymax, length = ndy) else 0.5 * (
				ymin + ymax)
		xd <- c(xd, rep(xt, ndy))
		yd <- c(yd, rep(yt, rep(ndx, ndy)))
	}
	ndm <- length(xd)
	list(x = c(x, xd), y = c(y, yd), ndm = ndm)
}

"extra.pts" <- 
function(xa, ya, deld = dout, eps = 0.0001)
{
	# Find barycentric coordinates for (ax,ay) points given D-Triangulation.
	# New points must lie inside convex hull of the D-Triangulation
	# It would be nice to have a better test for this!  Ideally it
	# would be good to have a function that returns whether each
	# extra point lies inside hull and then this could be used to trim
	# the new set of points, but this requires a stupid loop with chull
	# or some sort of clever new idea.
	na <- length(xa)
	hull <- chull(c(xa, deld$x), c(ya, deld$y))
	if(any(hull <= na))
		stop("extra points can't lie outside Delaunay hull")
	xout <- .Fortran("xtrapt",
		as.double(xa),
		as.double(ya),
		as.integer(na),
		as.integer(deld$nadj),
		as.integer(deld$madj),
		as.double(deld$x),
		as.double(deld$y),
		as.integer(deld$ntot),
		as.double(eps),
		iadd = integer(3 * na),
		badd = double(3 * na),
		nedge = integer(na),
		nerror = integer(1))
	if(xout$nerror > 0)
		stop(paste("Error", xout$nerror, "in xtrapt"))
	else {
		iadd <- t(matrix(xout$iadd, 3))
		badd <- t(matrix(xout$badd, 3))
		nedge <- xout$nedge
		return(iadd, badd, nedge)
	}
}

"fact" <- 
function(x)
gamma(x + 1)

"hcube" <- 
function(x, scale, translation)
{
	#   DATE WRITTEN:  24 April 1995          LAST REVISED:  1 May 1995
	#   AUTHOR:  Scott D. Chasalow
	#
	#   DESCRIPTION:
	#         Generate all points on a hypercuboid lattice. 
	#         Argument x is an integer vector giving the extent of each dimension; 
	#         the number of dimensions is length(x).  
	#         Argument scale is a vector of real numbers giving an amount by which 
	#         to multiply the points in each dimension;  it will be replicated as 
	#         necessary to have the same length as x.
	#         Argument translate is a vector of real numbers giving an amount to 
	#         translate (from the "origin", rep(1,length(x))) the points in each 
	#         dimension;  it will be replicated as necessary to have the same 
	#         length as x.  To use rep(0,length(x)) as the origin,  use 
	#         translation = -1.  Scaling,  if any,  is done BEFORE translation.
	#
	#   VALUE:
	#         A prod(x) by length(x) numeric matrix;  element (i,j) gives the 
	#         location of point i in the jth dimension.  The first column 
	#         (dimension) varies most rapidly.
	#
	#   SEE ALSO:
	#         fac.design,  expand.grid
	#
	ncols <- length(x)
	nrows <- prod(x)
	cp <- c(1, cumprod(x)[ - ncols])
	out <- lapply(as.list(1:length(x)), function(i, a, b, nr)
	rep(rep(1:a[i], rep(b[i], a[i])), length = nr), a = x, b = cp, nr = 
		nrows)
	out <- array(unlist(out), c(nrows, ncols))
	if(!missing(scale)) {
		scale <- rep(scale, length = ncols)
		out <- sweep(out, 2, scale, FUN = "*")
	}
	if(!missing(translation)) {
		translation <- rep(translation, length = ncols)
		out <- sweep(out, 2, translation, FUN = "+")
	}
	out
}

"ind.dup" <- 
function(x, y, rw = NULL, frac = 0.0001)
{
	#
	# Function ind.dup to calculate the indices of data pairs 
	# which duplicate earlier ones.  (Returns a logical vector;
	# true for such indices, false for the rest.)
	#
	if(is.null(rw)) rw <- c(0, 1, 0, 1)
	n <- length(x)
	rslt <- .Fortran("inddup",
		x = as.double(x),
		y = as.double(y),
		n = as.integer(n),
		rw = as.double(rw),
		frac = as.double(frac),
		dup = logical(n))
	rslt$dup
}

"logfact" <- 
function(x)
lgamma(x + 1)

"mipd" <- 
function(x, y)
{
	#
	# Function mipd to calculate the mean interpoint distance between
	# the points whose coordinates are stored in x and y.
	#
	n <- length(x)
	if(n != length(y))
		stop("data lengths do not match")
	.Fortran("mipd",
		x = as.double(x),
		y = as.double(y),
		n = as.integer(n),
		d = double(1))$d
}

"nCm" <- 
function(n, m, tol = 9.9999999999999986e-09)
{
	#  DATE WRITTEN:  7 June 1995               LAST REVISED:  10 July 1995
	#  AUTHOR:  Scott Chasalow
	#
	#  DESCRIPTION: 
	#        Compute the binomial coefficient ("n choose m"),  where n is any 
	#        real number and m is any integer.  Arguments n and m may be vectors;
	#        they will be replicated as necessary to have the same length.
	#
	#        Argument tol controls rounding of results to integers.  If the
	#        difference between a value and its nearest integer is less than tol,  
	#        the value returned will be rounded to its nearest integer.  To turn
	#        off rounding, use tol = 0.  Values of tol greater than the default
	#        should be used only with great caution, unless you are certain only
	#        integer values should be returned.
	#
	#  REFERENCE: 
	#        Feller (1968) An Introduction to Probability Theory and Its 
	#        Applications, Volume I, 3rd Edition, pp 50, 63.
	#
	len <- max(length(n), length(m))
	out <- numeric(len)
	n <- rep(n, length = len)
	m <- rep(m, length = len)
	mint <- (trunc(m) == m)
	out[!mint] <- NA
	out[m == 0] <- 1
	# out[mint & (m < 0 | (m > 0 & n == 0))] <-  0
	whichm <- (mint & m > 0)
	whichn <- (n < 0)
	which <- (whichm & whichn)
	if(any(which)) {
		nnow <- n[which]
		mnow <- m[which]
		out[which] <- ((-1)^mnow) * Recall(mnow - nnow - 1, mnow)
	}
	whichn <- (n > 0)
	nint <- (trunc(n) == n)
	which <- (whichm & whichn & !nint & n < m)
	if(any(which)) {
		nnow <- n[which]
		mnow <- m[which]
		foo <- function(j, nn, mm)
		{
			n <- nn[j]
			m <- mm[j]
			iseq <- seq(n - m + 1, n)
			negs <- sum(iseq < 0)
			((-1)^negs) * exp(sum(log(abs(iseq))) - lgamma(m + 1))
		}
		out[which] <- unlist(lapply(seq(along = nnow), foo, nn = nnow,
			mm = mnow))
	}
	which <- (whichm & whichn & n >= m)
	nnow <- n[which]
	mnow <- m[which]
	out[which] <- exp(lgamma(nnow + 1) - lgamma(mnow + 1) - lgamma(nnow -
		mnow + 1))
	nna <- !is.na(out)
	outnow <- out[nna]
	rout <- round(outnow)
	smalldif <- abs(rout - outnow) < tol
	outnow[smalldif] <- rout[smalldif]
	out[nna] <- outnow
	out
}

"nsimplex" <- 
function(p, n)
{
	# DATE WRITTEN:  24 Dec 1997 		 LAST REVISED:  24 Dec 1997
	# AUTHOR:  Scott D. Chasalow  (Scott.Chasalow@users.pv.wau.nl)
	#
	# DESCRIPTION:
	#       Computes the number of points on a {p, n}-simplex lattice; that is, the
	#	number of p-part compositions of n. This gives the number of points in
	#	the support space of a Multinomial(n, q) distribution, where
	#	p == length(q).
	#
	#	Arguments p and n are replicated as necessary to have the length of the
	#	longer of them.
	#
	# REQUIRED ARGUMENTS:
	#	p	vector of (usually non-negative) integers
	#	n	vector of (usually non-negative) integers
	# 
	mlen <- max(length(p), length(n))
	p <- rep(p, length = mlen)
	n <- rep(n, length = mlen)
	out <- nCm(n + p - 1, n)
	out[p < 0] <- 0
	out
}

"permn" <- 
function(x, fun = NULL, ...)
{
	# DATE WRITTEN: 23 Dec 1997          LAST REVISED:  23 Dec 1997
	# AUTHOR:  Scott D. Chasalow (Scott.Chasalow@users.pv.wau.nl)
	#
	# DESCRIPTION:
	#             Generates all permutations of the elements of x, in a minimal-
	#	change order. If x is a	positive integer,  returns all permutations
	#	of the elements of seq(x). If argument "fun" is not null,  applies
	#	a function given by the argument to each point. "..." are passed
	#	unchanged to the function given by argument fun, if any.
	#
	#	Returns a list; each component is either a permutation, or the
	#	results of applying fun to a permutation.
	#
	# REFERENCE:
	#	Reingold, E.M., Nievergelt, J., Deo, N. (1977) Combinatorial
	#	Algorithms: Theory and Practice. NJ: Prentice-Hall. pg. 170.
	#
	# SEE ALSO:
	#	sample, fact, combn, hcube, xsimplex
	#
	# EXAMPLE:
	#	# Convert output to a matrix of dim c(6, 720)
	#	t(array(unlist(permn(6)), dim = c(6, gamma(7))))
	#
	#	# A check that every element occurs the same number of times in each
	#	# position
	#	apply(t(array(unlist(permn(6)), dim = c(6, gamma(7)))), 2, tabulate, 
	#		nbins = 6)
	#
	#	# Apply, on the fly, the diff function to every permutation
	#	t(array(unlist(permn(6, diff)), dim = c(5, gamma(7))))
	#
	if(is.numeric(x) && length(x) == 1 && x > 0 && trunc(x) == x) x <- seq(
			x)
	n <- length(x)
	nofun <- is.null(fun)
	out <- vector("list", gamma(n + 1))
	p <- ip <- seqn <- 1:n
	d <- rep(-1, n)
	d[1] <- 0
	m <- n + 1
	p <- c(m, p, m)
	i <- 1
	use <-  - c(1, n + 2)
	while(m != 1) {
		out[[i]] <- if(nofun) x[p[use]] else fun(x[p[use]], ...)
		i <- i + 1
		m <- n
		chk <- (p[ip + d + 1] > seqn)
		m <- max(seqn[!chk])
		if(m < n)
			d[(m + 1):n] <-  - d[(m + 1):n]
		index1 <- ip[m] + 1
		index2 <- p[index1] <- p[index1 + d[m]]
		p[index1 + d[m]] <- m
		tmp <- ip[index2]
		ip[index2] <- ip[m]
		ip[m] <- tmp
	}
	out
}

"plot.deldir" <- 
function(object, add = F, wlines = c("both", "triang", "tess"), wpoints = c(
	"both", "real", "dummy", "none"), number = F, cex = 0.5, nex = 0.75,
	col = NULL, lty = NULL, pch = NULL, xlim = NULL, ylim = NULL)
{
	#
	# Function plot.deldir to produce a plot of the Delaunay triangulation
	# and Dirichlet tesselation of a point set, as produced by the
	# function deldir().
	#
	wlines <- match.arg(wlines)
	wpoints <- match.arg(wpoints)
	if(is.null(class(object)) || class(object) != "deldir") {
		cat("Argument is not of class deldir.\n")
		return(invisible())
	}
	if(is.null(col))
		col <- c(1, 1, 1, 1, 1)
	else col <- rep(col, length.out = 5)
	if(is.null(lty))
		lty <- 1:2
	else lty <- rep(lty, length.out = 2)
	if(is.null(pch))
		pch <- 1:2
	else pch <- rep(pch, length.out = 2)
	plot.del <- switch(wlines,
		both = T,
		triang = T,
		tess = F)
	plot.dir <- switch(wlines,
		both = T,
		triang = F,
		tess = T)
	plot.rl <- switch(wpoints,
		both = T,
		real = T,
		dummy = F,
		none = F)
	plot.dum <- switch(wpoints,
		both = T,
		real = F,
		dummy = T,
		none = F)
	delsgs <- object$delsgs
	dirsgs <- object$dirsgs
	n <- object$n.data
	rw <- object$rw
	if(plot.del) {
		x1 <- delsgs[, 1]
		y1 <- delsgs[, 2]
		x2 <- delsgs[, 3]
		y2 <- delsgs[, 4]
	}
	if(plot.dir) {
		u1 <- dirsgs[, 1]
		v1 <- dirsgs[, 2]
		u2 <- dirsgs[, 3]
		v2 <- dirsgs[, 4]
	}
	x <- object$summary[, 1]
	y <- object$summary[, 2]
	if(!add) {
		pty.save <- par()$pty
		on.exit(par(pty = pty.save))
		par(pty = "s")
		if(is.null(xlim))
			xlim <- rw[1:2]
		if(is.null(ylim))
			ylim <- rw[3:4]
		plot(0, 0, type = "n", xlim = xlim, ylim = ylim, xlab = "",
			ylab = "", axes = F)
		axis(side = 1)
		axis(side = 2)
		axes(xlab = "x", ylab = "y", axes = F)
	}
	if(plot.del)
		segments(x1, y1, x2, y2, col = col[1], lty = lty[1])
	if(plot.dir)
		segments(u1, v1, u2, v2, col = col[2], lty = lty[2])
	if(plot.rl) {
		x.real <- x[1:n]
		y.real <- y[1:n]
		points(x.real, y.real, pch = pch[1], col = col[3], cex = cex)
	}
	if(plot.dum) {
		x.dumm <- x[ - (1:n)]
		y.dumm <- y[ - (1:n)]
		points(x.dumm, y.dumm, pch = pch[2], col = col[4], cex = cex)
	}
	if(number) {
		xoff <- 0.02 * diff(range(x))
		yoff <- 0.02 * diff(range(y))
		text(x + xoff, y + yoff, 1:length(x), cex = nex, col = col[
			5])
	}
	invisible()
}

"print.deldir" <- 
function(x)
{
	#
	# Function print.deldir --- a ``method'' for print.
	# Needed only to keep quotes from being put into the output
	# in printing the ``dirseg'' component of the list returned
	# by deldir().
	#
	print.list(x, quote = F)
}

"rmultinomial" <- 
function(n, p, rows = max(c(length(n), nrow(p))))
{
	# 19 Feb 1997 (John Wallace, 17 Feb 1997 S-news)
	# Generate random samples from multinomial distributions, where both n
	# and p may vary among distributions
	#
	# Modified by Scott Chasalow
	#
	rmultinomial.1 <- function(n, p)
	{
		k <- length(p)
		tabulate(sample(k, n, replace = T, prob = p), nbins = k)
	}
	assign("rmultinomial.1", rmultinomial.1, frame = 1)
	n <- rep(n, length = rows)
	p <- p[rep(1:nrow(p), length = rows),  , drop = F]
	assign("n", n, frame = 1)
	assign("p", p, frame = 1)
	t(apply(matrix(1:rows, ncol = 1), 1, function(i)
	rmultinomial.1(n[i], p[i,  ])))
}

"rmultz2" <- 
function(n, p, draws = length(n))
{
	# 19 Feb 1997: From s-news 14 Feb 1997, Alan Zaslavsky
	# 11 Mar 1997: Modified by Scott D. Chasalow
	#
	# Generate random samples from a multinomial(n, p) distn: varying n, 
	# fixed p case.
	#
	n <- rep(n, length = draws)
	lenp <- length(p)
	tab <- tabulate(sample(lenp, sum(n), T, p) + lenp * rep(1:draws - 1,
		n), nbins = draws * lenp)
	dim(tab) <- c(lenp, draws)
	tab
}

"summary.rq" <- 
# This is a preliminary method for summarizing the output of the
# rq command eventually some bootstrapping strategies should be
# added.  In this instance, "summarizing" means essentially provision
# of either standard errors, or confidence intervals for the rq coefficents.
# Since the preferred method for confidence intervals is currently the
# rank inversion method available directly from rq() by setting ci=T, with br=T.
# these summary methods are intended primarily for comparison purposes
# and for use on large problems where the parametric programming methods
# of rank inversion are prohibitively memory/time consuming.  Eventually
# iterative versions of rank inversion should be developed that would
# employ the Frisch-Newton approach.  
#
# Object is the result of a call to rq(), and the function returns a
# table of coefficients, standard errors, "t-statistics", and p-values, and, if
# covariance=T a structure describing the covariance matrix of the coefficients, 
# i.e. the components of the Huber sandwich.
#
# There are three options for "se": 
#
#	1.  "iid" which presumes that the errors are iid and computes
#		an estimate of the asymptotic covariance matrix as in KB(1978).
#	2.  "nid" which presumes local (in tau) linearity (in x) of the
#		the conditional quantile functions and computes a Huber
#		sandwich estimate using a local estimate of the sparsity.
#	3.  "ker" which uses a kernel estimate of the sandwich as proposed
#		by Powell.
# See the inference chapter of the putative QR book for further details.
#
#
function(object, se = "nid", covariance = T)
{
	x <- object$x
	y <- object$y
	tau <- object$tau
	eps <- .Machine$single.eps
	wt <- object$weights
	coef <- coefficients(object)
	if(is.matrix(coef))
		coef <- coef[, 1]
	vnames <- dimnames(x)[[2]]
	resid <- object$residuals
	n <- length(resid)
	p <- length(coef)
	rdf <- n - p
	if(!is.null(wt)) {
		resid <- resid * wt
		x <- x * wt
		y <- y * wt
	}
	#quick and dirty se's in three flavors: iid, nid, and ker
	if(se == "iid") {
		xxinv <- diag(p)
		xxinv <- backsolve(qr(x)$qr[1:p, 1:p], xxinv)
		xxinv <- xxinv %*% t(xxinv)
		pz <- sum(abs(resid) < eps)
		h <- max(p + 1, ceiling(n * bandwidth.rq(tau, n, hs = T)))
		ir <- (pz + 1):(h + pz + 1)
		ord.resid <- sort(resid[order(abs(resid))][ir])
		sparsity <- l1fit(ir/(n - p), ord.resid)$coef[2]
		cov <- sparsity^2 * xxinv * tau * (1 - tau)
		serr <- sqrt(diag(cov))
	}
	else if(se == "nid") {
		h <- bandwidth.rq(tau, n, hs = T)
		bhi <- rq.fit.fn(x, y, tau = tau + h, int = F)$coef
		blo <- rq.fit.fn(x, y, tau = tau - h, int = F)$coef
		dyhat <- x %*% (bhi - blo)
		if(any(dyhat <= 0))
			warning(paste(sum(dyhat <= 0), "non-positive fis"))
		f <- pmax(0, (2 * h)/(dyhat - eps))
		fxxinv <- diag(p)
		fxxinv <- backsolve(qr(sqrt(f) * x)$qr[1:p, 1:p], fxxinv)
		fxxinv <- fxxinv %*% t(fxxinv)
		cov <- tau * (1 - tau) * fxxinv %*% crossprod(x) %*% fxxinv
		serr <- sqrt(diag(cov))
	}
	else if(se == "ker") {
		h <- qnorm(tau + bandwidth.rq(tau, n, hs = T))
		uhat <- y - x %*% coef
		f <- dnorm(uhat/h)/h
		fxxinv <- diag(p)
		fxxinv <- backsolve(qr(sqrt(f) * x)$qr[1:p, 1:p], fxxinv)
		fxxinv <- fxxinv %*% t(fxxinv)
		cov <- tau * (1 - tau) * fxxinv %*% crossprod(x) %*% fxxinv
		serr <- sqrt(diag(cov))
	}
	coef <- array(coef, c(p, 4))
	dimnames(coef) <- list(vnames, c("Value", "Std. Error", "t value",
		"Pr(>|t|)"))
	coef[, 2] <- serr
	coef[, 3] <- coef[, 1]/coef[, 2]
	coef[, 4] <- if(rdf > 0) 2 * (1 - pt(abs(coef[, 3]), rdf)) else NA
	object <- object[c("call", "terms", "assign")]
	if(covariance == T) {
		object$cov <- cov
		if(se != "iid") {
			object$Hinv <- fxxinv
			object$J <- crossprod(x)
		}
	}
	object$coefficients <- coef
	object$rdf <- rdf
	oldClass(object) <- "summary.rq"
	object
}

"triangles" <- 
function(x, y, H)
{
	#finds "other" vertices for edges of the Delaunay triangulation
	#
	#Input is original (x,y) points and delsgs matrix from deldir()
	#
	#Output is nby4 matrix consisting of
	#	first two columns from H containing indices of edge
	#	last two cols contain indices of other vertices of the edges triangles.
	#	if there are NA's in last two cols then these edges are on boundary.
	#
	#Strategy:  Find all the points which are connected to both ends of the edge
	#	if there are only two such points then we are done, otherwise
	#	compute signed area of all the candidate triangles and select
	#	the smallest negative and smallest positive ones
	#
	H <- H[, 5:6]
	K <- H + NA
	n <- nrow(H)
	for(k in 1:n) {
		i <- H[k, 1]
		j <- H[k, 2]
		Hi <- c(H[(H[, 1] == i) | (H[, 2] == i),  ])
		Hj <- c(H[(H[, 1] == j) | (H[, 2] == j),  ])
		Hij <- c(Hi, Hj)
		Hij <- Hij[is.na(match(Hij, c(i, j)))]
		Kij <- Hij[duplicated(Hij)]
		m <- length(Kij)
		if(m > 2) {
			area <- Kij
			for(h in 1:m) {
				verts <- c(i, j, Kij[h])
				area[h] <- tri.area(x[verts], y[verts])
			}
			if(min(area) > 0 | max(area) < 0)
				warning("can't find triangle")
			kp <- Kij[Kij == min(Kij[area > 0])]
			km <- Kij[Kij == max(Kij[area < 0])]
			if(length(Kij) > 2)
				browser
			Kij <- c(kp, km)
			m <- 2
		}
		K[k, 1:m] <- Kij
	}
	return(cbind(H, K))
}

"triogram" <- 
function(x, y, z, v = NULL, w = NULL, lambdas = 10^(-4:0), eps = 0.001, plotit
	 = F)
{
	# Fit experimental triogram model 
	#
	# This function fits a median (l1) nonparametric regression model to 
	# the observations (x,y,z).  The fitted function is piecewise linear
	# on triangles determined by a Delaunay triangulation of the (x[v],y[v])
	# points.  A vector of weights can be specified for the l1 fidelity
	# term.  This allows the vertices of the triangulation to be independent
	# of the data.  A vector of lambdas  may be specified to generate
	# a family of fits of varying smoothness controled by lambda.  The
	# smoothing penalty is the W(f) described in Koenker and Mizera (2000).
	#
	#
	# Version:  Sept 4, 1999
	# Input:
	#	(x,y,z)	data (and perhaps pseudo data corresponding to vertices.
	#	v	logical vector indicating vertices for D-Triangulation.
	#		Provisionally, we require that these v points include
	#		all points on the convex hull of the full set of pts.
	#	w	weight vector for l1 fidelity term.
	#	lambdas	vector of smoothing parameters for the fit.
	# Output:
	#	H is matrix of indices of Delaunay edges and adjacent vertices
	#	A is the matrix of the tv penalty:  
	#	W(f)= || A'z ||_1  total variation of interpolating fit for z.
	#
	# If first argument is a list, extract components x, y, and z.
	if(is.list(x)) {
		if(all(!is.na(match(c("x", "y", "z"), names(x))))) {
			z <- x$z
			y <- x$y
			x <- x$x
		}
		else stop("called with list lacking xyz elements")
	}
	N <- length(x)
	if(N != length(y) | N != length(z))
		stop("xyz lengths do not match")
	if(missing(v))
		v <- rep(T, N)
	if(missing(w))
		w <- rep(1, N)
	if(!is.logical(v))
		stop("v must be logical")
	if(any(w < 0))
		stop("w must be non-negative")
	if(N != length(v) | N != length(w))
		stop("vw lengths do not match")
	if(!any(v == T))
		stop("no vertices for triangulation")
	vx <- x[v]
	vy <- y[v]
	vz <- w[v] * z[v]
	dout <- deld(vx, vy, plotit = F)
	dind <- dout$ind
	npd <- dout$npd
	#number of parameters equal to number of vertices
	p <- length(vx)
	A <- w[v] * diag(p)
	G <- dout$delsgs
	# Make penalty contribution to design matrix.
	#
	#delete boundary edges
	if(plotit) {
		#force a square plotting region!
		par(pty = "s")
		plot(x, y, xlab = "x", ylab = "y", type = "n")
		segments(G[, 1], G[, 2], G[, 3], G[, 4], lty = 1)
	}
	if(any(v == F)) {
		ax <- x[!v]
		ay <- y[!v]
		az <- z[!v]
		aw <- w[!v]
		na <- length(ax)
		xout <- extra.pts(ax, ay, deld = dout)
		# Invert binning permutation: see Knuth, AoCP, v3 p13.
		indinv <- (1:npd)[order(dind)]
		K <- indinv[xout$iadd]
		K <- matrix(K, na, 3)
		# Delete extra points that fall outside convex hull 
		s <- apply(is.na(K), 1, sum) > 0
		az <- aw[!s] * az[!s]
		K <- K[!s,  ]
		B <- xout$badd
		B <- B[!s,  ]
		m <- nrow(K)
		a <- matrix(0, m, p)
		for(j in 1:3) {
			a[cbind(1:m, K[, j])] <- B[, j]
		}
		A <- rbind(A, aw[!s] * a)
	}
	H <- G[, 5:8]
	s <- apply(H < 1, 1, sum) == 0
	H <- H[s,  ]
	Gint <- G[s,  ]
	m <- nrow(H)
	b <- matrix(0, m, p)
	for(j in 1:4) {
		b[cbind(1:m, H[, j])] <- Gint[, j + 8]
	}
	# Make response vector
	#
	if(any(v == F)) z <- c(vz, az, rep(0, m)) else z <- c(vz, rep(0, m))
	n <- length(z) - m
	nlam <- length(lambdas)
	coef <- matrix(0, p, nlam)
	fid <- rep(0, nlam)
	pen <- rep(0, nlam)
	plam <- rep(0, nlam)
	sic <- rep(0, nlam)
	for(j in 1:nlam) {
		lambda <- lambdas[j]
		D <- rbind(A, lambda * b)
		fit <- rq(z ~ D - 1, tau = 0.5, ci = F)
		fid[j] <- sum(abs(fit$resid[1:n]))
		pen[j] <- sum(abs(fit$resid[(n + 1):(n + m)]))/lambda
		plam[j] <- sum(abs(fit$resid[1:n]) < eps)
		sic[j] <- (0.5 * log(n) * plam[j])/n + log(fid[j]/(2 * (n -
			plam[j])))
		coef[, j] <- fit$coef
	}
	return(coef, G, fid, pen, plam, sic)
}

"tv.penalty" <- 
function(x, y, z, xadd = 0, yadd = 0, mu = 0.10000000000000001, plotxy = F)
{
	#Compute experimental tv penalty matrix for piecewise linear funs on triangles.
	#For each edge of the triangulation:
	#	compute 2norm of  difference in gradient vector for adjacent triangles.
	#	sum these  -- weighted by the lengths of the edges
	#	edges that are on the boundary are ignored...caveat emptor!
	#This is W(F) where F is piecewise linear on the triangles.
	#Rolf Turner's Delaunay Package is used to compute the triangulation.
	#
	#N.B.  the z's (fun values at the vertices) aren't really needed except for
	#debugging purposes.  
	#
	#Version:  May 16, 1999
	#Input:
	#	(x,y,z)
	#Output:
	#	H is the matrix of indices of the delaunay edges and adjacent vertices
	#	A is the matrix of the tv penalty:  
	#	TV= || A'z ||_1  total variation of interpolating fit for z.
	#
	G <- deldir(x, y, xadd, yadd, plotit = T)$delsgs
	#force a square plotting region!
	par(pty = "s")
	#delete boundary edges
	if(plotxy) {
		plot(x, y, xlab = "x", ylab = "y", type = "n")
		segments(G[, 1], G[, 2], G[, 3], G[, 4], lty = 1)
	}
	H <- G[, 5:8]
	s <- apply(H < 1, 1, sum) == 0
	H <- H[s,  ]
	G <- G[s,  ]
	m <- nrow(H)
	p <- length(x)
	A <- matrix(0, m, p)
	A <- matrix(0, m, p)
	for(j in 1:4) {
		A[cbind(1:m, H[, j])] <- G[, j + 8]
	}
	return(A, TV = sum(abs(A %*% z)))
}

"tv.triangles" <- 
function()
{
}

"x2u" <- 
function(x, labels = seq(along = x))
{
	#  DATE WRITTEN:  21 January 1994       LAST REVISED:  21 January 1994
	#  AUTHOR:  Scott Chasalow
	#
	#  DESCRIPTION:
	#        Convert an x-encoded simplex-lattice point to a u-encoded
	#        simplex-lattice point  (equivalently,  "untabulate" bin counts)
	#
	#  USAGE:
	#        x2u(x)
	#
	#  ARGUMENTS:
	#  x:    A numeric vector.  x[i] is interpreted as the count in bin i.
	#  labels:  A vector.  Interpreted as the bin labels;  default value is
	#        seq(along = x), which causes return of a u-encoded simplex-lattice 
	#        point.  Other values of labels cause return of the result of 
	#        subscripting labels with the u-encoded simplex-lattice point that 
	#        would have been obtained if the default value of labels were used.
	#
	#        Arguments x and labels must be of equal length.
	#
	#  VALUE:
	#        rep(labels, x), a vector of length sum(x).  If labels = seq(along = x)
	#        (the default),  value is the u-encoded translation of the simplex 
	#        lattice point, x.  Equivalently,  value gives the bin numbers, 
	#        in lexicographic order,  for the objects represented by the counts in 
	#        x.  For other values of argument "labels", value gives the bin labels 
	#        for the objects represented by the counts in x (equivalent to 
	#        labels[x2u(x)]).
	#
	#  SEE ALSO:
	#        tabulate,  rep
	#
	if(length(labels) != length(x)) stop(
			"Arguments x and labels not of equal length")
	rep(labels, x)
}

"xsimplex" <- 
function(p, n, fun = NULL, simplify = T, ...)
{
	#       DATE WRITTEN: 11 February 1992          LAST REVISED:  10 July 1995
	#       AUTHOR:  Scott Chasalow
	#
	#       DESCRIPTION:
	#             Generates all points on a {p,n} simplex lattice (i.e. a p-part 
	#             composition of n).  Each point is represented as x, a 
	#             p-dimensional vector of nonnegative integers that sum to n.
	#             If argument "fun" is not null,  applies a function given
	#             by the argument to each point.  If simplify is FALSE,  returns 
	#             a list; else returns a vector or an array.  "..." are passed 
	#             unchanged to function given by argument fun,  if any.
	#       EXAMPLE:
	#             Compute Multinomial(n = 4, pi = rep(1/3, 3)) p.f.:
	#             xsimplex(3, 4, dmnom, prob=1/3) 
	#
	if(p < 1 || n < 0) return(if(simplify) numeric(0) else list())
	p1 <- p - 1
	x <- numeric(p)
	x[1] <- n
	nofun <- is.null(fun)
	out <- if(nofun) x else fun(x, ...)
	if(p == 1 || n == 0) {
		return(if(simplify) out else list(out))
	}
	count <- nCm(n + p - 1, n)
	if(simplify) {
		dim.use <- NULL
		if(nofun) {
			if(count > 1)
				dim.use <- c(p, count)
		}
		else {
			d <- dim(out)
			if(count > 1) {
				if(length(d) > 1)
					dim.use <- c(d, count)
				else if(length(out) > 1)
					dim.use <- c(length(out), count)
			}
			else if(length(d) > 1)
				dim.use <- d
		}
	}
	out <- vector("list", count)
	target <- 1
	i <- 0
	while(1) {
		i <- i + 1
		out[[i]] <- if(nofun) x else fun(x, ...)
		x[target] <- x[target] - 1
		if(target < p1) {
			target <- target + 1
			x[target] <- 1 + x[p]
			x[p] <- 0
		}
		else {
			x[p] <- x[p] + 1
			while(x[target] == 0) {
				target <- target - 1
				if(target == 0) {
					i <- i + 1
					out[[i]] <- if(nofun) x else fun(x,
							...)
					if(simplify) {
						if(is.null(dim.use))
							out <- unlist(out)
						else out <- array(unlist(out),
								dim.use)
					}
					return(out)
				}
			}
		}
	}
}

