#' @title Read Citation Data
#' @description Read citation data from Clarivate Journal Citation Reports
#' @param jname journal name
#' @param dir journal directory, Default: 'Stat'
#' @param type Citation type, Default: 'Cited'
#' @return Matrix of journal citation counts
#' @details Journal names are extracted by \code{journals}, each journal
#' has two files "Cited" and "Citing".  Data directory is either Econ or Stat.
#' @details Extract citation data for specified journals
#' @rdname JCRead
#' @export 
JCRead <- function(jname, dir = "Stat", type = "Cited"){
    fpath <- system.file("extdata", package = "RRpairwise")
    dir <- paste(fpath, "/", dir, sep = "")
    file <- Sys.glob(paste(dir, "/", jname, "*",type, "*", sep = ""))
    B <- read.csv(file, header = FALSE, skip = 8)[, 3:4]
    B[,1] <- gsub(" ","-",B[,1])
    dimnames(B)[[2]] <- c("Journal", "Cite")
    B
}
#' @title Extract Journal Names
#' @description Extract unique set of journal names
#' @return list of unique econ and stat journals.
#' There are a few journals that appear in both lists/directories
#' so duplicate names are discarded.
#' @details Extract the names of all the journals
#' @examples 
#' \dontrun{
#' if(interactive()){
#'  #EXAMPLE1
#'  }
#' }
#' @rdname journals
#' @export 
journals <- function(){
    fpath <- system.file("extdata", package = "RRpairwise")
    fecon <- list.files(paste(fpath, "/Econ", sep = ""))
    fstat <- list.files(paste(fpath, "/Stat", sep = ""))
    jecon <- unique(gsub("#.*","", fecon))
    jstat <- unique(gsub("#.*","", fstat))
    jstat <- jstat[!(jstat %in% jecon)]
    list(jecon = jecon, jstat = jstat)
}
#' @title Citation Matrices
#' @description Construct matrix of citation counts
#' @param type Cited or Citing matrix, Default: 'Cited'
#' @return Square matrix of citation counts
#' @details The Cited and Citing matrices should be transposes of one
#' another, however the JC files don't quite reflect this, the example
#' code illustrates this discrepancy.
#' @examples 
#' Cited <- Cites()
#' Citing <- Cites(type = "Citing")
#' # Sanity check:  Data in the Cited and Citing matrices should match.
#' # They do not, but the discrepancies are modest.
#' A <- data.matrix(Cited)
#' B <- data.matrix(Citing)
#' C0 <- all.equal(A,t(B))
#' C1 <- sum(A != t(B))
#' mn <- which(A != t(B))
#' D <- cbind(mn %/% 86, 1 + mn %% 86, A[A != t(B)], t(B)[A != t(B)])
#' @rdname Cites
#' @export 
Cites <- function(type = "Cited"){
    js <- journals()
    alljs <- sort(c(js$jecon, js$jstat))
    njs <- length(alljs)
    dirs <- c("Econ", "Stat")
    D <- data.frame(Journal = alljs)
    for(i in 1:2){
	for(j in js[[i]]){
	    B <- JCRead(j, dir = dirs[i], type = type)
	    Bnz <- B[B[,1] %in% alljs,]
	    nnz <- nrow(Bnz)
	    j0s <- alljs[!(alljs %in% Bnz[,1])]
	    Bz <- data.frame(Journal = j0s, Cite = rep(0,njs - nnz))
	    B <- rbind(Bnz, Bz)
	    B <- B[order(B[,1]),]
	    D <- cbind(D, B[,2])
	}
    }
    cnames <- c(js[[1]],js[[2]])
    o <- order(cnames)
    G <- D[,-1]
    G <- G[,o]
    dimnames(G) <- list(D[,1], cnames[o])
    G
}

