#' Interpolation function for bivariate Renyi fitting
#'
#' Bivariate Renka interpolation for Renyi fitting
#'
#' This is an interpolation procedure intended for contour plotting
#' or surface visualization via rgl, or some other schema.  Given
#' function values of a transformed density \eqn{y = g(x)} at scattered points 
#' \eqn{x}, as produced by \code{medde2}, satisfying  a convexity constraint, 
#' we want to find interpolated values of \eqn{g} on a lattice \eqn{p_x \times p_y}.
#' The algorithm and implementation is based on Renka's (2004) CSRFPACK, which is
#' unfortunately licensed under the TOMS "for non-commercial use" scheme, and consequently 
#' cannot appear on CRAN.
#'
#' @param x original data points (n by 2 matrix)
#' @param y fitted g function values at x points
#' @param px grid of x values for interpolation
#' @param py grid of y values for interpolation
#' @param tolbe tolerance parameter for csurf
#' @return A matrix of interpolated function values
#' @references Renka RJ. 2004. Algorithm 833: {CSRFPACK} interpolation of 
#' scattered data with a {C1} convexity-preserving surface. \emph{ACM TOMS} 
#' 30: 200 -- 211.
#' @author Roger Koenker
#' @keywords nonparametric
#' @useDynLib MeddeR
#' @export
#' 

ggrid <- function(x, y, px, py, tolbe = 1e-4) {
    # compute function values on a grid  px x py
    nx <- length(px)
    ny <- length(py)
    n <- length(y)
    s <- .Fortran("csurf",
		  as.integer(n),
		  as.double(x[,1]),
		  as.double(x[,2]),
		  as.double(tolbe),
		  as.integer(0),
		  as.double(1),
		  strict = as.logical(FALSE),
		  nr = as.integer(8),
		  na = as.integer(24),
		  nd = integer(1),
		  y = as.double(y),
		  c = double(n),
		  tlist = integer(6*n-12),
		  tlptr = integer(6*n-12),
		  tlend = integer(n),
		  tlnew = integer(1),
		  tnear = integer(n),
		  tnext = integer(n),
		  nv = integer(1),
		  tlistv = integer(6*n-12),
		  dxl = double(2*n-2),
		  dyl = double(2*n-2),
		  gx = double(n),
		  gy = double(n),
		  eps = as.double(0),
		  d = double(n),
		  dmin = double(1),
		  w = double(8),
		  qx = double(8*24),
		  qy = double(8*24),
		  ier = integer(1), PACKAGE = "MeddeR")
    if(s$ier != 0)
         stop(paste("csurf failed:  ier = ", s$ier))
    g <- .Fortran("fgrid",
		  as.integer(nx),
		  as.integer(ny),
		  as.double(px),
		  as.double(py),
		  as.double(s$eps),
		  as.double(s$gx),
		  as.double(s$gy),
		  as.double(s$c),
		  as.integer(s$tlist),
		  as.integer(s$tlptr),
		  as.integer(s$tlend),
		  as.double(s$dmin),
		  as.integer(s$nr),
		  as.integer(s$na),
		  as.double(s$w),
		  as.double(s$qx),
		  as.double(s$qy),
		  f = double(nx*ny),
		  ier = integer(1), PACKAGE = "MeddeR")
    if(g$ier != 0)
         stop(paste("fgrid failed:  ier = ", g$ier))
    matrix(g$f,nx,ny)
}
