cqr.fit.LNA <- function(X, Z, y, delta, g, taus, link, h = NULL){
    n <- length(y)
    lam <- rep(1,n)
    Tau <- max(y * delta)
    B <- NWweight(Z, h, lam)
    G <- gfit(Tau, y, delta, Z, B, g$coef, lam)
    pZg <- plogis(Z %*% G$g)
    beta <- bfit(y, delta, Z, weights = pZg)
    list(gamma = G$g, beta = coef(beta, taus))
}
cqr.fit.Imp <- function(X, Z, y, delta, g, taus, link, grid = 1:99/100, h = NULL){
    n <- length(y)
    lam <- rep(1,n)
    Tau <- max(y * delta)
    B <- NWweight(Z, h, lam)
    G <- gfit(Tau, y, delta, Z, B, g$coef, lam)
    pZg <-plogis(Z %*% G$g)
    beta <- impute(y, delta, X, B, S= diag(G$S), lam, weights = pZg, grid = grid)
    list(gamma = G$g, beta = coef(beta, taus))
}
cqr.fit.DA <- function(X, Z, y, delta, link, grid = 1:99/100, B = NULL, gam=NULL, 
	   taus, L=NULL, U=NULL, h = NULL, bootstrap = TRUE, maxit = 50, Large = 1e4){
    n <- length(y)
    if(is.null(B)){
	uncen <- which(delta == 0)
	f <- rq(y ~ X - 1, tau=0.5, subset = uncen)   
	r <- quantile(y[uncen] - (X[uncen,-1, drop = FALSE] %*% f$coef[-1]), probs = grid)
	B <- cbind(r, outer(r/r, f$coef[-1]))  # parallel slopes
    }
    if(is.null(gam))
	gam = glm(delta ~ Z - 1, binomial(link = link))$coef
    if(is.null(L)) L = rep(-Large,n) 
    if(is.null(U)) {
      cen = which(delta==0) 
      U = 0*L + Large 
      U[cen] = y[cen]
    }
    G <- array(0, c(ncol(X), length(taus), maxit)) 
    H <- matrix(0, ncol(Z), maxit)
    h <- 0
    while (h < maxit){
	h <- h + 1
	eta <- Draweta(X, y, delta, Z, B, grid, gam)
	gam <-Drawgam (Z, eta,bootstrap)
	H[,h] <- gam
	uncured <- which(eta==1)
	yh <- Drawy(X[uncured,], y[uncured], delta[uncured], U[uncured], L[uncured], B)
	B <- DrawB(yh, X[uncured,],grid,bootstrap)
	if(sum(B^2) < 10000 * length(grid)) 
	    G[,,h] <- rq(yh ~ X[uncured,] - 1, tau = taus)$coef
	else G[,,h] <- matrix(NA, ncol(X),length(taus))
    }
    list(beta = apply(G, 1:2, mean), gamma = apply(H, 1, mean))
} 

# Beta estimation procedure for Imputation/Data Augmentation given the logistic weights
impute <-function(y, D, X, B, S, lam, nimp = 5, weights, grid = NULL){
    if(is.null(grid))
	grid <- seq(0,1,by=min(0.01,1/(2*length(y)^.7)))
    if (any(weights < 0)) stop("negative weights")
    KMW <- function(a,b) 
	ifelse( (b-1+a > 0 & a >0), (b-1+a)/a, 1) 
    p <- ncol(X)
    n <- length(y)
    m <- length(grid)
    Tau <- max(y * D)
    tmp <- weights * S
    hatp <- D + (1 - D) * tmp/ifelse(1 - weights + tmp == 0, 1, 1 - weights + tmp)
    hatp <- pmax(hatp, 1e-6)
    hbet <- matrix(0,p,m)
    xbar <- apply(X, 2, mean)
    for(i in 1:nimp){ 
	eta = rbinom(n,1,hatp)
	omeF <- outer(S,grid, KMW)
	omeF[which(D==1),] <- 1
	uncr <- which(eta == 1)
	kmweights <- omeF[uncr,]; 
	y.uncr <- y[uncr]; D.uncr <- D[uncr]; X.uncr <- X[uncr,];lam.uncr <- lam[uncr]
	# censored objects among uncured
	ncen <- sum(D.uncr == 0)
	y.aug <- rep(1e6, ncen) ## Points augmented at infinity as in Portnoy's CRQ
	X.aug <-  X.uncr[D.uncr==0,] 
	kmweights <- lam.uncr*kmweights
	kmweights.aug <- lam.uncr[D.uncr==0]*(1-kmweights[D.uncr ==0,])
	ynew <- c(y.uncr,y.aug)  
	Xnew <- rbind(X.uncr,X.aug)
	wts <- rbind(kmweights,kmweights.aug) 
	hbet <- hbet + sapply((1:m), function(j) rq(ynew ~ 0+Xnew,tau=grid[j],weights = wts[,j])$coef)
   }
   beta <- hbet/nimp
   Qhat <- t(xbar) %*% beta
   beta <- rbind(grid,beta, Qhat)
   dimnames(beta) <- list(c("tau",dimnames(X)[[2]],"Qhat"),NULL)
   class(beta) <- "cqr"
   beta
}
coef.cqr <- function (object, taus = 1:4/5, ...) {
    if (min(taus) < 0 || max(taus) > 1) 
        stop("taus out of range [0,1]")
    taus <- sort(taus)
    S <- object
    r <- S[1, ]
    r <- c(r[1], r)
    if (is.unsorted(r)) 
        r <- r[-length(r)]
    B <- S[-1, , drop = FALSE]
    B <- t(cbind(B, B[, ncol(B), drop = FALSE]))
    ts <- taus[taus > min(r) & taus < max(r)]
    bin <- findInterval(ts, r)
    wgt <- (ts - r[bin])/(r[bin + 1] - r[bin])
    binlag <- bin - 1
    binlag[binlag == 0] <- 1
    coef <- t(wgt * B[bin, , drop = FALSE] + 
	 (1 - wgt) * B[binlag, , drop = FALSE])
    nna <- length(taus) - length(ts)
    if (nna > 0) 
        coef <- cbind(coef, matrix(NA, nrow(coef), nna))
    taulabs <- paste("tau=", format(round(taus, 3)))
    dimnames(coef)[[2]] <- taulabs
    coef[-nrow(coef), ]
}

## DArq for Cure CRQ:
Drawy <- function(X, y, delta, U, L, B){
    # Only allows right censoring...
    cen <- which(delta == 0)
    if(length(cen)){
    for (j in 1:length(cen)) {
	yj <- B %*% X[cen[j],] 
	if (min(yj) < L[cen[j]]) {
           lo <- which(yj < L[cen[j]])
           if (length(lo) != 1) 				
              y[cen[j]] <- yj[sample(lo, 1)]
	   else
              y[cen[j]] <- yj[lo]
  	   }
	}
    }
    y
}
    Draweta <- function(X, y, delta, Z, B,grid = 1:99/100, gam){
    n <- length(y)
    wts <- plogis(c(Z %*% gam))
    fit <- X %*% t(B)
    ## Computes the quantile level  where X falls based on initial estimator of beta
    indvec <- sapply((1:n), function(u) round(rank(c(y[u],fit[u,]))[1]))
    egrid <- c(grid,1) 
    qsurv <- (1 - egrid[indvec])
    tmp <- wts*qsurv
    hatp <- delta+(1-delta)*tmp/(1-wts+tmp)
    hatp <- ifelse(hatp>=0, hatp, 1e-6)
    rbinom(n, 1, hatp)
}

Drawgam <- function(Z, eta, bootstrap){
    n = length(eta)
    if (bootstrap) { 
	s <- sample(1:n, n, replace=TRUE)
	gam <- as.vector(glm(eta[s]~ 0 + Z[s,], binomial)$coef)
    }
    else 
	gam = as.vector(glm(eta ~ Z - 1,binomial)$coef)
    gam
}

DrawB <- function(yh, X,grid,bootstrap){
    n = length(yh)
    if (bootstrap) { 
	s <- sample(1:n, n, replace=TRUE)
	B <- t(rq(yh[s] ~ 0+X[s,], tau=grid)$coef)
    }  
    else
	B <- t(rq(yh ~ 0+X, tau = grid)$coef)
    B
}
biweight <- function(x) (15/16) * (abs(x) <= 1) * (1 - x^2)^2
NWweight <- function(Z, h, Lam, K = biweight){
	n <- NROW(Z)
	p <- NCOL(Z)
	B <- apply(Z, 2, function(x) outer(x,x,"-"))
	B <- array(B, c(n,n,p))
	for(i in 1:p) B[,,i] <- K(B[,,i]/h[i])
	B <- apply(B, 1:2, prod)
	B/apply(B,1,sum)
}
LocalKM <- function(y, D, B, tau, omega = NULL, eta = NULL){
	# Either omega OR eta should be passed, but NOT both
	# Imputation employs the eta's, otherwise omega's.
	n <- length(y)
	o <- order(y)
	y <- y[o]
	D <- D[o]
	B <- B[o,o]
	if(length(eta)) B <- eta[o] * B
	N <- B %*% (D * outer(y, y, '<='))
	dN <- N - cbind(0, N[,-n])
	if(length(omega)) B <- omega[o,o] * B
	R <- B %*% outer(y, y, '>=')
	S <- 1 - dN/R
	S[is.nan(S)] <- 1
	S <- apply(S, 1, cumprod)
	S[y >= tau,] <- 0
	return(S[order(o),order(o)])
}
Omega <- function(D, Z, g, S){
	# This is (5,7) of Wu-Yin (2016)
	p <- plogis(c(Z %*% g))
	pS <- p * t(S)
	t(D + (1 - D) * pS/(1 - p + pS))
}
gfit <- function(tau, y, D, Z, B, g0, Lam, cntl = list(btol = 0.01)){
    n <- nrow(Z)
    p <- ncol(Z)
    omega <- matrix(1,n,n)
    S <- LocalKM(y, D, B, tau, omega)
    EEg <- function(g){
	p <- plogis(c(Z %*% g))
	R <- p * (1 - diag(S))
	R <- Z * Lam * (1 - p) * (D - R)/(1 - R)
	apply(R,2,sum)
    }
    gap <- 1
    while(gap > 1e-2){
	omega <- Omega(D, Z, g0, S)
	S <- LocalKM(y, D, B, tau, omega)
	fitg <- nleqslv(g0, EEg, control = cntl)
	eflag <- fitg$termcd
	if(eflag != 1) break
	gap <- max(abs(g0 - fitg$x))
	g0 <- fitg$x
    }
    omega <- Omega(D, Z, g0, S)
    S <- LocalKM(y, D, B, tau, omega)
    list(g = g0, S = S, eflag = eflag)
}
#### Martingale approach to estimate Beta given the logistic weights
#### This is the non-iterative approach of Wu and Yin (2013)
bfit <- function(y,D,X, weights=NULL, 
		 grid = seq(0,1,by=min(0.01,1/(2*length(y)^.7)))) {
    p <- ncol(X)
    n <- length(y)
    m <- length(grid)
    xbar <- apply(X,2,mean)
      if (any(weights < 0)) stop("negative weights")
    dH <- t(apply(-log(1 - weights %*% t(grid)), 1, diff))
    uhat <- rep(1,n)
    s <- rep(0,n)
    B <- matrix(0,p,m)
    y1 <- y[(D==1)]
    x1 <- X[(D==1),]
    for(j in 1:(m-1)){
	s <- s + (uhat >= 0)* dH[,j]
	rhs <- t(X) %*% (D - s)
	b <- try(rq.fit.fnd(x1,y1,rhs)$coef,TRUE)
	if(is.vector(b))
	   B[,j] <- b
	else break
	uhat <- y - X %*% B[,j]
    }
    qhat <- t(xbar) %*% B
    B <- rbind(grid,B,qhat)[,1:(j-1)]
    dimnames(B) <- list(c("tau",dimnames(X)[[2]],"Qhat"),NULL)
    class(B) <- "cqr"
    B
}
rq.fit.fnd <- function (x, y,  rhs, beta = 0.99995, eps = 1e-06) {
    n <- length(y)
    p <- ncol(x)
    if (n != nrow(x)) 
        stop("x and y don't match n")
    d <- rep(0, n)
    u <- rep(1, n)
    wn <- rep(0, 10 * n)
    wn[1:n] <- .5 
    z <- .Fortran("rqfnb", as.integer(n), as.integer(p), a = as.double(t(as.matrix(x))), 
        c = as.double(-y), rhs = as.double(rhs), d = as.double(d), 
        as.double(u), beta = as.double(beta), eps = as.double(eps), 
        wn = as.double(wn), wp = double((p + 3) * p), it.count = integer(3), 
        info = integer(1), PACKAGE = "quantreg")
    if (z$info != 0) 
        stop(paste("Error info = ", z$info, "in stepy: singular design"))
    coefficients <- -z$wp[1:p]
    names(coefficients) <- dimnames(x)[[2]]
    list(coefficients = coefficients)
}
