#' Estimate a voronogram density or regression function
#' 
#' Voronograms are piecewise constant functions defined on a Voronoi
#' tesselation of the plane.  Estimation of voronograms is done by minimizing
#' an l1 measure of fidelity regularized by a penalty defined by the total
#' variation of the estimated density.
#' 
#' The function estimates either a piecewise constant  quantile regression function if the argument
#' \code{z} is provided, or a density function if \code{z} is missing.  In the
#' latter case, the argument \code{tau} is formally valid, but may be hard to
#' interpret.  For density estimation the function minimizes an l1 measure of
#' fidelity subject to a constraint on the total variation of fitted density.
#' The reference distribution for the fidelity criterion is given by the
#' voronogram that put mass 1/n in each of the Voronoi cells. By default
#' weights are chosen in the density estimation setting so that the fidelity
#' can be interpreted as a functional version of the l1 distance between the
#' fitted and reference density.  The example below illustrates the density
#' functionality, for the regression version try:  \code{demo(tseg)}.
#' 
#' @aliases voronogram 
#' @param x observations on \code{x}
#' @param y observations on \code{y}
#' @param z observations on \code{z}
#' @param wgts weights
#' @param lambda regularization parameter
#' @param tau quantile parameter
#' @return list of class voronogram with components: 
#'	\item{x}{observations on \code{x} as used in the fitting}
#' 	\item{y}{observations on \code{y} as used in the fitting}
#' 	\item{z}{observations on \code{z} as used in the fitting} 
#'	\item{fit}{fitted function values }
#' @note This function draws heavily on the tripack package for various
#' 	functionality for producing and manipulating Voronoi tesselations.
#' @author Roger Koenker
#' @seealso \code{\link{plot.voronogram}}
#' @keywords smooth
#' @importFrom methods as new
#' @export
#' @examples
#' x <- rnorm(500)
#' y <- rnorm(500)
#' vm <- tripack::voronoi.mosaic(x,y)
#' g <- vareas(x,y,vm)
#' v <- voronogram(x,y,lambda=.5)
#' plot(v)
#' 
"voronogram" <- function(x,y,z,wgts = rep(1,length(x)), lambda=1, tau = .5){
	n <- length(x)
	vm <- tripack::voronoi.mosaic(x,y)
	constraint <- FALSE
	if(missing(z)){
		v <- vareas(x,y,vm)
		x <- v$xx
		y <- v$yy
		z <- v$zz
		vm <- v$vm
		wgts <- c(1/z[1:n],rep(1,length(z)-n))
		r <- c(1,-1)
		w <- c(1/z[1:n],rep(0,length(z) - n))
		R <- SparseM::as.matrix.csr(rbind(w, -w)/n)
		n <- length(x)
		constraint = TRUE
		}
	B <- vedges(vm)
	dumx <- B$dumx
	dumy <- B$dumy
	vx <- c(vm$x,dumx)
	vy <- c(vm$y,dumy)
	B <- B$A
        elen <- sqrt((vx[B[1,]]-vx[B[2,]])^2 + (vy[B[1,]]-vy[B[2,]])^2)
	m <- ncol(B)
	ra <-  c(rbind(elen,-elen)) 
	ja <- as.integer(c(B[3:4,])) 
	ia <- as.integer(1 + (0:m)*2) 
	dim  <- as.integer( c(m,n))
        H <- new("matrix.csr",ra=ra,ja=ja,ia=ia,dimension=dim)
	X <- rbind(wgts*as(n,"matrix.diag.csr"),lambda*H)
	zz <- c(wgts*z,rep(0,m))
	if(constraint){
	    cntl <- quantreg::sfn.control(nnzlmax = 64 * length(X@ra))
	    fit <- quantreg::rq.fit.sfnc(X, zz, R, r, tau = .5, control = cntl)
	}
	else
		fit <- quantreg::rq.fit.sfn(X,zz,tau = .5)
	if(fit$ierr != 0)warning(paste("convergence problem: ierr = ",fit$ierr))
	fit <- fit$coef
	jumps <- H %*% fit
	Z <- list(x =x, y = y, z = z, fit=fit)
	class(Z) <- "voronogram"
	Z
        }
