#' Voronoi Edges
#'
#' Find the Voronoi edges and the associated data points for each of them.
#'
#' @param z is a Voronoi object, ie output of voronoi.mosaic from library(tripack)
#'
#' @return An object of class vedges with components:
#' 	\item{A}{a matrix with 4 rows first two give indices of edge vertices
#' 		second two rows give the indices of the associated data points:
#' 		If the indices in the first two rows are negative then they index a set of
#' 		dummy vertices that lie on the intersection of the voronoi edge and the
#' 		convex hull of the observations.}
#' 	\item{dumx}{dummy x vertices}
#' 	\item{dumy}{dummy y vertices}
#' @export
"vedges" <-
function(z){
	m <- length(z$x)
	A <- matrix(0,4,3*m)
	k <- 0
	inhull <- tripack::in.convex.hull(z$tri,z$x,z$y)
	onhull <- tripack::convex.hull(z$tri)$i
	onhull <- c(onhull,onhull[1])
	dumx <- rep(0,m)
	dumy <- rep(0,m)
	dumn <- 0
	for(v0 in 1:m){
		t0 <- c(z$p1[v0], z$p2[v0], z$p3[v0])
		for(i in 1:3){
			v1 <- z[[paste("n",i,sep="")]][v0]
			vd0 <- v0
			vd1 <- v1
			if(v1 < 0 || v0 < 0){
				if(v0>0){
					vin  <- v0
					vout <- -v1
					}
				else if(v1>0){
					vin  <- v1
					vout <- -v0
					}
				else
					stop(paste(c(v0,v1),"outside hull"))
				for(j in 1:(length(onhull)-1)){
                                  x0 <- c(z$x[vin],z$tri$x[onhull[j]])
                                  y0 <- c(z$y[vin],z$tri$y[onhull[j]])
                                  x1 <- c(z$dummy.x[vout],z$tri$x[onhull[j+1]])
                                  y1 <- c(z$dummy.y[vout],z$tri$y[onhull[j+1]])
                                  pt <- point.of.intersection(x0,y0,x1,y1)
                                  if(length(pt)>1) {
                                                dumn <- dumn+1
                                                dumx[dumn] <- pt$x
                                                dumy[dumn] <- pt$y
                                                vd0 <- vin
                                                vd1 <- m + dumn
                                                }
                                        }
                                }
			else if(v1 > v0){
			  if(inhull[v0] && inhull[v1])
				point.in.hull <- TRUE
			  else {
	  		  	point.in.hull <- FALSE
				for(j in 1:(length(onhull)-1)){
					x0 <- c(z$x[v0],z$tri$x[onhull[j]])
					y0 <- c(z$y[v0],z$tri$y[onhull[j]])
					x1 <- c(z$x[v1],z$tri$x[onhull[j+1]])
					y1 <- c(z$y[v1],z$tri$y[onhull[j+1]])
					pt <- point.of.intersection(x0,y0,x1,y1)
					if(length(pt)>1) {
						point.in.hull <- TRUE
						dumn <- dumn+1
						dumx[dumn] <- pt$x
						dumy[dumn] <- pt$y
						if(!inhull[v0])vd0 <- m+dumn
						if(!inhull[v1])vd1 <- m+dumn
						}
					}
				}
			if(point.in.hull){
				k <- k+1
				t1 <- c(z$p1[v1], z$p2[v1], z$p3[v1])
				if(length(intersect(t0,t1))==2)
					A[,k] <- c(vd0,vd1,intersect(t0,t1))
				}
			}
		}
	}
A <- A[,1:k]
dumx <- dumx[1:dumn]
dumy <- dumy[1:dumn]
Z <- list(A=A, dumx=dumx, dumy=dumx)
class(Z) <- "vedges"
Z
}
