#' Point of Intersection
#'
#' Given two line segments Determine whether they intersect, 
#' if so return point of intersection
#'
#' @param x0 initial x coordinate
#' @param y0 initial y coordinate
#' @param x1 end x coordinate
#' @param y1 end y coordinate
#'
#' @return FALSE if segments don't intersect, else
#'      \item{xint}{point of intersection}
#'      \item{yint}{point of intersection}
#' @export
"point.of.intersection" <- function(x0,y0,x1,y1) {
    ccw <- function(p0,p1,p2){
	dx1 = p1$x - p0$x
	dy1 = p1$y - p0$y
	dx2 = p2$x - p0$x 
	dy2 = p2$y - p0$y
	if (dx1*dy2 > dy1*dx2) return(+1)
	if (dx1*dy2 < dy1*dx2) return(-1)
	if ((dx1*dx2 < 0) || (dy1*dy2 < 0)) return(-1)
	if ((dx1*dx1+dy1*dy1) < (dx2*dx2+dy2*dy2)) 
					    return(+1)
	return(0)
    }
    segments.intersect <- function(l1,l2) {
	return(((ccw(l1$p1, l1$p2, l2$p1)
		*ccw(l1$p1, l1$p2, l2$p2)) <= 0)
	    && ((ccw(l2$p1, l2$p2, l1$p1)
		*ccw(l2$p1, l2$p2, l1$p2)) <= 0))
    }
    l1 <- list(p1=list(x=x0[1],y=y0[1]), p2=list(x=x1[1],y=y1[1]))
    l2 <- list(p1=list(x=x0[2],y=y0[2]), p2=list(x=x1[2],y=y1[2]))
    if(!segments.intersect(l1,l2)) return(FALSE)
    else {
        b1 <- (y1[1]-y0[1])/(x1[1]-x0[1])
        b2 <- (y1[2]-y0[2])/(x1[2]-x0[2])
        a1 <- y0[1] - b1*x0[1]
        a2 <- y0[2] - b2*x0[2]
        xint <- (a2 - a1)/(b1 - b2)
        yint <- (b1*a2 - a1*b2)/(b1 - b2)
        }
    list(xint=xint,yint=yint)
}
