# Functions for simulating, plotting  and predicting the outcome of an NCAA type single 
# elimination tournament

#	First version:  Feb 25, 2007  RK
#	Revised version:  March 20, 2014  RK
#	Revised version:  March 24, 2014  RK
#	Revised version:  February 6, 2015  RK

Bracket <- function(QBFit, Tournament) {
    Fit <- QBFit$coef
    copula <- QBFit$copula
    seeds <- Tournament$seeds
    slots <- Tournament$slots
    teams <- Tournament$teams
    Nteams <- Tournament$Nteams
    # Simulate a full realization of the NCAA tournament
    Matchups <- function(R, seeds){
      R[,1] <- seeds[match(R[,1], seeds[,2]), 3]
      R[,2] <- seeds[match(R[,2], seeds[,2]), 3]
      R
      }
   NCAARound <- function(games, Fit, Nteams, copula = frankCopula(2.5, dim=2)){
   # This function simulates one round of the NCAA tournament
   #
   # G is a g by 2 matrix of team numbers for the g games in the round
   # Fit is the matrix of fitted QR coefs for the estimated model
   # NTeams is the vector of team names for the fitted model for matching
   # copula is the function that simulates draws from the estimated copula model
   # Returns winning teams and their team id number for the round.
	g <- rep(0,nrow(games))
	J <- dim(Fit)[2] + 1
	G <- games
	G[,1] <- factor(Nteams[match(games[,1],Nteams[,1]),2], Nteams[,2])
	G[,2] <- factor(Nteams[match(games[,2],Nteams[,1]),2], Nteams[,2])
	H <- model.matrix(~G[,2] - 1)[,-1,drop=FALSE]
	A <- model.matrix(~G[,1] - 1)[,-1,drop=FALSE]
	X <- rbind(cbind(H,-A),cbind(A,-H))
	X <- cbind(1,0,X)
	R <- X %*% Fit
	W <- rCopula(nrow(G), copula)
	W[,1] <- pmin(pmax(1,round(W[,1]*J)),J-1)
	W[,2] <- pmin(pmax(1,round(W[,2]*J)),J-1)
	w <- c(W)
	S <- matrix(diag(R[,w]),nrow(G),2)[,2:1,drop=FALSE]
	g <- ifelse((S[,1] - S[,2]) > 0,G[,1],G[,2])
        scores = data.frame(G[,1], S[,1], G[,2], S[,2])
	list(winners = Nteams[g,], scores = scores)
	}
    # Preliminary Round
    W <- list()
    R <- slots[nchar(slots$slot) == 3,3:4]
    games <- Matchups(R, seeds)
    W[[1]] <- w <- NCAARound(games, Fit, Nteams)
    seeds <- seeds[-match(setdiff(unlist(games),w$winners[,1]),seeds[,3]),]
    seeds[,2] <- gsub("[ab]","",seeds[,2])

    # Remaining Rounds 
    for(i in 1:4){
	k <- 2^(3:0)
	WXYZ <- LETTERS[23:26]
	R <- slots[grep(paste("R",i,sep = ""),slots$slot),3:4]
        games <- Matchups(R, seeds)
        W[[i+1]] <- w <- NCAARound(games, Fit, Nteams)
        losers <- match( setdiff(unlist(games),w$winners[,1]), seeds[,3])
        seeds <- seeds[-losers,] 
	seeds[,3] <- w$winners[,1]
	seeds[,2] <- paste("R", i, rep(WXYZ, each = k[i]), rep(1:k[i],4), sep = "")
    }

    # Semi-final Round
    R <- slots[grep("R5",slots$slot),3:4]
    games <- Matchups(R, seeds)
    W[[6]] <- w <- NCAARound(games, Fit, Nteams, copula = copula)
    losers <- match( setdiff(unlist(games),w$winners[,1]) ,seeds[,3])
    seeds[losers,3] <- seeds[match(w$winners[,1], seeds[,3]),3] # maintain order!
    seeds <- seeds[-losers,]
    seeds[,2] <- paste("R5", c("WX","YZ"), sep = "")
    # Final Round
    R <- slots[grep("R6",slots$slot),3:4]
    games <- Matchups(R, seeds)
    W[[7]] <- NCAARound(games, Fit, Nteams)
    class(W) <- "Bracket"
    W
}
			

plot.Bracket <- function(x, ...){
    # Fill in a simulated bracket 
    # x contains outcome of the simulation
    # E.g.  pdf(file, height=8, width=12)
  BracketLayout <- function(){
     grid.newpage()
     pushViewport(viewport(width=.9,height=.9))
     pushViewport(viewport(layout = grid.layout(2,10)))
  }

  PlotRound <- function(Teams,Seeds,Scores, Reflect = FALSE){
      ngames <- length(Teams)/2
      nround <- 4 - log2(ngames)
      offset <- c(0,3,9,21)
      skip <- c(0,6,18,0)
      col <- (nround - 1) * 26
      r <- 1
      if(Reflect) {r <- -1; col <- 104 - col}
      for(k in 1:ngames){
        for(j in 1:2){
	   i <-  j +(k-1)*2
	   row <-  4 + (k-1)*6 + (j-1)*2 + offset[nround]  + skip[nround]*(k-1)
	   #Seed Box only for first round
	   if(nround == 1){
              pushViewport(viewport(layout.pos.col = col + r * (1:3), layout.pos.row = row:(row+1)))
              grid.rect(gp=gpar(col="gray", fill="cornsilk"))
              grid.text(Seeds[i],gp=gpar(fontsize=8, col=gray(0.1)))
              popViewport()
	   }
	   #Name Box
           pushViewport(viewport(layout.pos.col = col + r * (4:21), layout.pos.row = row:(row+1)))
           grid.rect(gp=gpar(col="gray", fill="cornsilk"))
	   if(j == 1 && (Scores[i] > Scores[i+1]))
              grid.text(Teams[i],gp=gpar(fontsize=8, fontface="bold",col=gray(0.1)))
	   else if(j == 2 && (Scores[i] > Scores[i-1]))
              grid.text(Teams[i],gp=gpar(fontsize=8, fontface="bold",col=gray(0.1)))
	   else 
	      grid.text(Teams[i],gp=gpar(fontsize=8, col=gray(0.1)))
           popViewport()
	   #Score Box
           pushViewport(viewport(layout.pos.col = col + r * (21:23), layout.pos.row = row:(row+1)))
           grid.rect(gp=gpar(col="gray", fill="cornsilk"))
           grid.text(Scores[i],gp=gpar(fontsize=8, col=gray(0.1)))
           popViewport()
           }
       }
   }
  PlotFinal4 <- function(Teams, Scores){
   pushViewport(viewport(layout.pos.row = 1:2, layout.pos.col = 5:6))
   pushViewport(viewport(layout = grid.layout(100,56)))
   ngames <- length(Teams)/2
   final <- (ngames == 1)
   for(k in 1:ngames){
       for(j in 1:2){
	   i <- j + 2 * (k-1)
	   rowTeam <- 50:51
	   colTeam <- (k - 1) * 28 + 1 + (4:21) 
	   colScore <- (k - 1) * 28 + 1 + (21:23) 
	   if(final) {
	       colTeam <-  14 + (4:21) 
	       colScore <- 14 + (21:23) 
	       rowTeam <- 40:41
	   }
	   if(j == 2) rowTeam <- rowTeam + 2
	   #Name Box
           pushViewport(viewport(layout.pos.col = colTeam, layout.pos.row = rowTeam))
           grid.rect(gp=gpar(col="gray", fill="cornsilk"))
	   if(j == 1 && (Scores[i] > Scores[i+1]))
              grid.text(Teams[i],gp=gpar(fontsize=8, fontface="bold",col=gray(0.1)))
	   else 
	      grid.text(Teams[i],gp=gpar(fontsize=8, col=gray(0.1)))
           popViewport()
	   #Score Box
           pushViewport(viewport(layout.pos.col = colScore, layout.pos.row = rowTeam))
           grid.rect(gp=gpar(col="gray", fill="cornsilk"))
           grid.text(Scores[i],gp=gpar(fontsize=8, col=gray(0.1)))
           popViewport()
       }
   }
   popViewport()
}


  RegionalBracketLayout <- function(Region){
   regions <- matrix(c("East","South","West","Midwest"), 2, 2)
   quadrant <- which(regions == Region, arr.ind = TRUE)
   if(quadrant[2] == 1) cols <- 1:4
   else cols <- 7:10
   pushViewport(viewport(layout.pos.row = quadrant[1], layout.pos.col = cols))
   pushViewport(viewport(layout = grid.layout(50,104)))
   pushViewport(viewport(layout.pos.col = 1:104, layout.pos.row = 1:2))
   grid.rect(gp=gpar(col="gray", fill="lightblue"))
   grid.text(paste("NCAA", Region))
   popViewport()
}
    BracketLayout()
    regions <- c("East","South","Midwest","West")
    Seeds <- c(1,16,8,9,5,12,4,13,6,11,3,14,7,10,2,15)
    for(i in 1:4){
	region <- regions[i]
	Reflect <- region %in% c("West", "Midwest")
        RegionalBracketLayout(region)
	S <- x[[2]]$scores
	m <- nrow(S)/4
	S <- S[(i-1) * m + 1:m,]
	S <- S[c(1,8,5,4,6,3,7,2),]
	Teams <- c(t(S[,c(1,3)]))
	Scores <- floor(c(t(S[,c(2,4)])))
        for(j in 3:6){
            PlotRound(Teams,Seeds,Scores,Reflect = Reflect)
	    S <- x[[j]]$scores
	    m <- nrow(S)/4
	    S <- S[(i-1) * m + 1:m,]
	    if(j == 3) S <- S[c(1,4,3,2),]
	    Teams <- c(t(S[,c(1,3)]))
	    Scores <- floor(c(t(S[,c(2,4)])))
	}
       popViewport()
       popViewport()
    }
    for(i in 6:7){
       S <- x[[i]]$scores
       Teams <- c(t(S[,c(1,3)]))
       Scores <- floor(c(t(S[,c(2,4)])))
       PlotFinal4(Teams, Scores)
       popViewport()
    }
}
